<?php
require_once '../config.php';
requireRole('admin');

$backup_dir = __DIR__ . '/../backups/';
$success_message = '';
$error_message = '';

// --- Handle Restore Request ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['restore_database'])) {
    if (isset($_FILES['backup_file']) && $_FILES['backup_file']['error'] == 0) {
        $file = $_FILES['backup_file'];
        $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);

        if (strtolower($file_extension) !== 'sql') {
            $error_message = "Invalid file type. Please upload a .sql file.";
        } else {
            $sql_dump = file_get_contents($file['tmp_name']);
            
            if ($sql_dump === false) {
                $error_message = "Could not read the uploaded backup file.";
            } else {
                try {
                    $db = new Database();
                    $conn = $db->getConnection();
                    
                    // Temporarily disable foreign key checks to avoid errors during restore
                    $conn->exec('SET foreign_key_checks = 0');

                    // Execute the SQL dump
                    $conn->exec($sql_dump);
                    
                    // Re-enable foreign key checks
                    $conn->exec('SET foreign_key_checks = 1');

                    logActivity($_SESSION['user_id'], 'Database restored from backup', "File: " . htmlspecialchars($file['name']));
                    $success_message = "Database successfully restored from " . htmlspecialchars($file['name']) . ".";

                } catch (PDOException $e) {
                    $error_message = "Database restore failed: " . $e->getMessage();
                }
            }
        }
    }
}

// --- Handle File Deletion/Download ---
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['file'])) {
    $file = basename($_GET['file']);
    $filepath = $backup_dir . $file;

    if (file_exists($filepath)) {
        if (isset($_GET['action']) && $_GET['action'] === 'delete') {
            if (unlink($filepath)) {
                logActivity($_SESSION['user_id'], 'Deleted database backup', "File: $file");
                $success_message = "Successfully deleted backup file: $file";
            } else {
                $error_message = "Could not delete backup file: $file";
            }
        } elseif (isset($_GET['action']) && $_GET['action'] === 'download') {
            logActivity($_SESSION['user_id'], 'Downloaded database backup', "File: $file");
            header('Content-Description: File Transfer');
            header('Content-Type: application/octet-stream');
            header('Content-Disposition: attachment; filename="' . $file . '"');
            header('Expires: 0');
            header('Cache-Control: must-revalidate');
            header('Pragma: public');
            header('Content-Length: ' . filesize($filepath));
            readfile($filepath);
            exit;
        }
    } else {
        $error_message = "File not found: $file";
    }
}

// --- List Backup Files ---
$backup_files = glob($backup_dir . "*.sql");
rsort($backup_files); // Sort by most recent first

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Database Backups - Admin - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: #f7d2f7ff;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
        }
        .card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0,0,0,0.12);
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-database me-2"></i>Database Backups</h1>
                </div>

                <?php if ($success_message): ?><div class="alert alert-success"><?php echo $success_message; ?></div><?php endif; ?>
                <?php if ($error_message): ?><div class="alert alert-danger"><?php echo $error_message; ?></div><?php endif; ?>

                <div class="card">
                    <div class="card-header">
                        <h5>Available Backups</h5>
                    </div>
                    <div class="card-body">
                        <p>Backups are generated automatically once per day. You can download or delete them from here. Backups older than 30 days are automatically removed.</p>
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Backup Date</th>
                                        <th>File Size</th>
                                        <th class="text-end">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($backup_files)): ?>
                                        <tr><td colspan="3" class="text-center text-muted">No backups found.</td></tr>
                                    <?php else: ?>
                                        <?php foreach ($backup_files as $file): ?>
                                            <tr>
                                                <td><?php echo basename($file); ?></td>
                                                <td><?php echo round(filesize($file) / 1024, 2); ?> KB</td>
                                                <td class="text-end">
                                                    <a href="?file=<?php echo basename($file); ?>&action=download" class="btn btn-sm btn-primary">
                                                        <i class="fas fa-download me-1"></i> Download
                                                    </a>
                                                    <a href="?file=<?php echo basename($file); ?>&action=delete" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this backup?');">
                                                        <i class="fas fa-trash-alt me-1"></i> Delete
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <div class="card-footer text-muted">
                        <i class="fas fa-info-circle me-2"></i>To trigger a backup manually for testing, visit the cron script URL directly.
                    </div>
                </div>

                <!-- Restore from Backup -->
                <div class="card mt-4 border-danger">
                    <div class="card-header bg-danger text-white">
                        <h5 class="mb-0"><i class="fas fa-exclamation-triangle me-2"></i>Restore Database from Backup</h5>
                    </div>
                    <div class="card-body">
                        <p class="text-danger"><strong>Warning:</strong> This is a destructive action. Restoring from a backup will completely overwrite the current database with the data from the selected file. All data entered after the backup was created will be permanently lost.</p>
                        <p>Only proceed if you are absolutely sure. It is highly recommended to create a new backup of the current state before restoring.</p>
                        <hr>
                        <form method="POST" enctype="multipart/form-data" onsubmit="return confirm('Are you absolutely sure you want to overwrite the entire database? This cannot be undone.');">
                            <div class="mb-3">
                                <label for="backup_file" class="form-label">Select Backup File (.sql)</label>
                                <input class="form-control" type="file" id="backup_file" name="backup_file" accept=".sql" required>
                            </div>
                            <button type="submit" name="restore_database" class="btn btn-danger">
                                <i class="fas fa-upload me-2"></i>Restore Database
                            </button>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>