<?php
require_once '../config.php';
require_once '../vendor/autoload.php';
requireRole('admin');

$db = new Database();
$conn = $db->getConnection();

// Get statistics
$stats = [];
$stmt = $conn->query("SELECT COUNT(*) as total FROM users WHERE role = 'patient'");
$stats['patients'] = $stmt->fetchColumn();

$stmt = $conn->query("SELECT COUNT(*) as total FROM users WHERE role = 'doctor'");
$stats['doctors'] = $stmt->fetchColumn();

$stmt = $conn->query("SELECT COUNT(*) as total FROM follicle_records");
$stats['records'] = $stmt->fetchColumn();

$stmt = $conn->query("SELECT COUNT(*) as total FROM activity_logs WHERE DATE(created_at) = CURDATE()");
$stats['today_activities'] = $stmt->fetchColumn();

// Get subscription stats for doughnut chart
$stmt_subs = $conn->query("
    SELECT 
        SUM(CASE WHEN subscription_status = 'active' THEN 1 ELSE 0 END) as active,
        SUM(CASE WHEN subscription_status = 'expired' THEN 1 ELSE 0 END) as expired,
        SUM(CASE WHEN subscription_status = 'inactive' OR subscription_status IS NULL THEN 1 ELSE 0 END) as inactive
    FROM patients
");
$subscription_stats = $stmt_subs->fetch(PDO::FETCH_ASSOC);

// Recent activities
$stmt = $conn->prepare("SELECT al.*, u.username FROM activity_logs al JOIN users u ON al.user_id = u.id ORDER BY al.created_at DESC LIMIT 10");
$stmt->execute();
$recent_activities = $stmt->fetchAll();

// --- Chart Data: New Subscriptions in Last 30 Days ---
$new_subs_data = [];
$stmt_new_subs = $conn->query("
    SELECT DATE(created_at) as subscription_date, COUNT(*) as count
    FROM activity_logs
    WHERE action = 'Subscription activated'
      AND created_at >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
    GROUP BY DATE(created_at)
    ORDER BY subscription_date ASC
");
$new_subs_by_date = $stmt_new_subs->fetchAll(PDO::FETCH_KEY_PAIR);

$chart_labels = [];
$chart_data = [];
for ($i = 29; $i >= 0; $i--) {
    $date = date('Y-m-d', strtotime("-$i days"));
    $chart_labels[] = date('M j', strtotime($date));
    $chart_data[] = $new_subs_by_date[$date] ?? 0;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: #f8f9fa;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
        }
        .card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0,0,0,0.12);
        }
    </style>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Admin Dashboard</h1>
                </div>
                
                <!-- Statistics Cards -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card bg-primary text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h5 class="card-title">Patients</h5>
                                        <h2><?php echo $stats['patients']; ?></h2>
                                    </div>
                                    <i class="fas fa-users fa-2x"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-success text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h5 class="card-title">Doctors</h5>
                                        <h2><?php echo $stats['doctors']; ?></h2>
                                    </div>
                                    <i class="fas fa-user-md fa-2x"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-info text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h5 class="card-title">Records</h5>
                                        <h2><?php echo $stats['records']; ?></h2>
                                    </div>
                                    <i class="fas fa-file-medical fa-2x"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-warning text-white">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h5 class="card-title">Today's Activities</h5>
                                        <h2><?php echo $stats['today_activities']; ?></h2>
                                    </div>
                                    <i class="fas fa-chart-line fa-2x"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Charts -->
                <div class="row mb-4">
                    <div class="col-md-8">
                        <div class="card h-100">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-chart-line me-2"></i>New Subscriptions (Last 30 Days)</h5>
                            </div>
                            <div class="card-body">
                                <canvas id="newSubscriptionsChart"></canvas>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card h-100">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-chart-pie me-2"></i>Subscription Status</h5>
                            </div>
                            <div class="card-body"><canvas id="subscriptionStatusChart"></canvas></div>
                        </div>
                    </div>
                </div>
                
                <!-- Recent Activities -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Recent Activities</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>User</th>
                                        <th>Action</th>
                                        <th>Details</th>
                                        <th>IP Address</th>
                                        <th>Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($recent_activities as $activity): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($activity['username']); ?></td>
                                        <td><?php echo htmlspecialchars($activity['action']); ?></td>
                                        <td><?php echo htmlspecialchars($activity['details'] ?? '-'); ?></td>
                                        <td><?php echo htmlspecialchars($activity['ip_address']); ?></td>
                                        <td><?php echo date('M j, Y H:i', strtotime($activity['created_at'])); ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Subscription Status Doughnut Chart
            const statusCtx = document.getElementById('subscriptionStatusChart').getContext('2d');
            new Chart(statusCtx, {
                type: 'doughnut',
                data: {
                    labels: ['Active', 'Expired', 'Inactive'],
                    datasets: [{
                        label: 'Subscription Status',
                        data: [
                            <?php echo $subscription_stats['active'] ?? 0; ?>,
                            <?php echo $subscription_stats['expired'] ?? 0; ?>,
                            <?php echo $subscription_stats['inactive'] ?? 0; ?>
                        ],
                        backgroundColor: [
                            'rgba(40, 167, 69, 0.7)',  // Success
                            'rgba(220, 53, 69, 0.7)',  // Danger
                            'rgba(108, 117, 125, 0.7)' // Secondary
                        ],
                        borderColor: [
                            'rgba(40, 167, 69, 1)',
                            'rgba(220, 53, 69, 1)',
                            'rgba(108, 117, 125, 1)'
                        ],
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            position: 'bottom',
                        }
                    }
                }
            });

            // New Subscriptions Line Chart
            const newSubsCtx = document.getElementById('newSubscriptionsChart').getContext('2d');
            new Chart(newSubsCtx, {
                type: 'line',
                data: {
                    labels: <?php echo json_encode($chart_labels); ?>,
                    datasets: [{
                        label: 'New Subscriptions',
                        data: <?php echo json_encode($chart_data); ?>,
                        borderColor: 'rgba(13, 110, 253, 1)',
                        backgroundColor: 'rgba(13, 110, 253, 0.1)',
                        fill: true,
                        tension: 0.3
                    }]
                },
                options: {
                    responsive: true,
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: { stepSize: 1 }
                        }
                    }
                }
            });
        });
    </script>
</body>
</html>