<?php
require_once '../config.php';
requireRole('admin');

$db = new Database();
$conn = $db->getConnection();

// --- Filtering and Search Logic ---
$search_term = sanitize($_GET['search'] ?? '');

$where_clauses = ["r.name = 'doctor'"];
$params = [];

if ($search_term !== '') {
    $where_clauses[] = "(u.username LIKE ? OR u.email LIKE ?)";
    $params[] = "%$search_term%";
    $params[] = "%$search_term%";
}

$where_sql = 'WHERE ' . implode(' AND ', $where_clauses);

// --- Pagination Logic ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 20;
$offset = ($page - 1) * $limit;

// Get total records for pagination
$count_sql = "
    SELECT COUNT(u.id) 
    FROM users u
    JOIN user_roles ur ON u.id = ur.user_id
    JOIN roles r ON ur.role_id = r.id
    $where_sql
";
$count_stmt = $conn->prepare($count_sql);
$count_stmt->execute($params);
$total_records = $count_stmt->fetchColumn();
$total_pages = ceil($total_records / $limit);

// --- Fetch Data ---
$sql = "
    SELECT u.id, u.username, u.email, u.status, u.created_at, COUNT(p.id) as patient_count
    FROM users u
    JOIN user_roles ur ON u.id = ur.user_id
    JOIN roles r ON ur.role_id = r.id
    LEFT JOIN patients p ON u.id = p.doctor_id
    $where_sql
    GROUP BY u.id
    ORDER BY u.created_at DESC
    LIMIT $limit OFFSET $offset
";

$stmt = $conn->prepare($sql);
$stmt->execute($params);
$doctors = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Doctors - Admin - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: #f8f9fa;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
        }
        .card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0,0,0,0.12);
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-user-md me-2"></i>Manage Doctors</h1>
                </div>

                <!-- Search Form -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3 align-items-end">
                            <div class="col-md-9">
                                <label for="search" class="form-label">Search Doctor</label>
                                <input type="text" class="form-control" id="search" name="search" placeholder="Username or Email" value="<?php echo htmlspecialchars($search_term); ?>">
                            </div>
                            <div class="col-md-3">
                                <button type="submit" class="btn btn-primary w-100"><i class="fas fa-search me-2"></i>Search</button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Doctors Table -->
                <div class="card">
                    <div class="card-header">
                        <h5>Doctor List (Showing <?php echo count($doctors); ?> of <?php echo $total_records; ?>)</h5>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th>Username</th>
                                        <th>Email</th>
                                        <th>Status</th>
                                        <th>Assigned Patients</th>
                                        <th>Registered On</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($doctors)): ?>
                                        <tr>
                                            <td colspan="6" class="text-center text-muted p-4">No doctors found.</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($doctors as $doctor): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($doctor['username']); ?></td>
                                                <td><?php echo htmlspecialchars($doctor['email']); ?></td>
                                                <td>
                                                    <span class="badge bg-<?php echo $doctor['status'] == 'active' ? 'success' : 'secondary'; ?>">
                                                        <?php echo ucfirst($doctor['status']); ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <span class="badge bg-primary"><?php echo $doctor['patient_count']; ?></span>
                                                </td>
                                                <td><?php echo date('M j, Y', strtotime($doctor['created_at'])); ?></td>
                                                <td>
                                                    <a href="users.php?search=<?php echo urlencode($doctor['username']); ?>" class="btn btn-sm btn-outline-primary" title="Manage User">
                                                        <i class="fas fa-user-edit"></i> Manage
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <?php if ($total_pages > 1): ?>
                    <div class="card-footer">
                        <!-- Pagination -->
                        <nav>
                            <ul class="pagination justify-content-center mb-0">
                                <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search_term); ?>">Previous</a>
                                </li>
                                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                    <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search_term); ?>"><?php echo $i; ?></a>
                                    </li>
                                <?php endfor; ?>
                                <li class="page-item <?php echo $page >= $total_pages ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search_term); ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>