<?php
require_once '../config.php';
requireRole('admin');

$db = new Database();
$conn = $db->getConnection();

$success_message = '';
$error_message = '';

// --- Handle Form Submissions ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // --- Create a new role ---
        if (isset($_POST['create_role'])) {
            $role_name = sanitize($_POST['role_name']);
            if (!empty($role_name)) {
                $stmt = $conn->prepare("INSERT INTO roles (name) VALUES (?)");
                $stmt->execute([$role_name]);
                $success_message = "Role '{$role_name}' created successfully.";
                logActivity($_SESSION['user_id'], "Created role: {$role_name}");
            } else {
                $error_message = "Role name cannot be empty.";
            }
        }

        // --- Update an existing role and its permissions ---
        if (isset($_POST['update_role'])) {
            $role_id = (int)$_POST['role_id'];
            $role_name = sanitize($_POST['role_name']);
            $permissions = $_POST['permissions'] ?? [];

            if (!empty($role_name) && $role_id > 0) {
                // Begin transaction
                $conn->beginTransaction();

                // 1. Update role name
                $stmt = $conn->prepare("UPDATE roles SET name = ? WHERE id = ?");
                $stmt->execute([$role_name, $role_id]);

                // 2. Delete existing permissions for this role
                $stmt = $conn->prepare("DELETE FROM role_permissions WHERE role_id = ?");
                $stmt->execute([$role_id]);

                // 3. Insert new permissions
                if (!empty($permissions)) {
                    $stmt = $conn->prepare("INSERT INTO role_permissions (role_id, permission_id) VALUES (?, ?)");
                    foreach ($permissions as $permission_id) {
                        $stmt->execute([$role_id, (int)$permission_id]);
                    }
                }

                // Commit transaction
                $conn->commit();
                $success_message = "Role '{$role_name}' updated successfully.";
                logActivity($_SESSION['user_id'], "Updated role: {$role_name} (ID: {$role_id})");
            } else {
                $error_message = "Invalid role data provided for update.";
            }
        }

        // --- Delete a role ---
        if (isset($_POST['delete_role'])) {
            $role_id = (int)$_POST['role_id'];
            $stmt = $conn->prepare("SELECT name FROM roles WHERE id = ?");
            $stmt->execute([$role_id]);
            $role_name = $stmt->fetchColumn();

            // Safeguard against deleting critical roles
            if (in_array(strtolower($role_name), ['admin', 'doctor', 'patient'])) {
                $error_message = "Cannot delete core system role '{$role_name}'.";
            } else {
                $stmt = $conn->prepare("DELETE FROM roles WHERE id = ?");
                $stmt->execute([$role_id]);
                $success_message = "Role '{$role_name}' deleted successfully.";
                logActivity($_SESSION['user_id'], "Deleted role: {$role_name} (ID: {$role_id})");
            }
        }
    } catch (PDOException $e) {
        if ($conn->inTransaction()) {
            $conn->rollBack();
        }
        $error_message = "Database error: " . $e->getMessage();
    }
}

// --- Fetch Data for Display ---
$stmt_roles = $conn->query("SELECT * FROM roles ORDER BY name");
$roles = $stmt_roles->fetchAll();

$stmt_permissions = $conn->query("SELECT * FROM permissions ORDER BY name");
$all_permissions = $stmt_permissions->fetchAll();

// Create a map of role_id to its permissions
$role_permissions_map = [];
$stmt_role_perms = $conn->query("SELECT role_id, permission_id FROM role_permissions");
while ($row = $stmt_role_perms->fetch()) {
    $role_permissions_map[$row['role_id']][] = $row['permission_id'];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Roles & Permissions - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: #f8f9fa;
        }
        .card, .accordion-item {
            border: none;
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
        }
        .card:hover, .accordion-item:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0,0,0,0.12);
        }
    <style>
        .permission-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 1rem;
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>

            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-user-shield me-2"></i>Manage Roles & Permissions</h1>
                </div>

                <?php if ($success_message): ?>
                    <div class="alert alert-success alert-dismissible fade show"><?php echo $success_message; ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div>
                <?php endif; ?>
                <?php if ($error_message): ?>
                    <div class="alert alert-danger alert-dismissible fade show"><?php echo $error_message; ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div>
                <?php endif; ?>

                <!-- Create New Role Form -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="fas fa-plus-circle me-2"></i>Create New Role</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" class="row g-3 align-items-end">
                            <div class="col-md-6">
                                <label for="role_name" class="form-label">Role Name</label>
                                <input type="text" class="form-control" id="role_name" name="role_name" required>
                            </div>
                            <div class="col-md-6">
                                <button type="submit" name="create_role" class="btn btn-primary">Create Role</button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Existing Roles Accordion -->
                <div class="accordion" id="rolesAccordion">
                    <?php foreach ($roles as $role): ?>
                        <div class="accordion-item">
                            <h2 class="accordion-header" id="heading-<?php echo $role['id']; ?>">
                                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapse-<?php echo $role['id']; ?>">
                                    <strong><?php echo htmlspecialchars($role['name']); ?></strong>
                                </button>
                            </h2>
                            <div id="collapse-<?php echo $role['id']; ?>" class="accordion-collapse collapse" data-bs-parent="#rolesAccordion">
                                <div class="accordion-body">
                                    <form method="POST">
                                        <input type="hidden" name="role_id" value="<?php echo $role['id']; ?>">
                                        
                                        <!-- Edit Role Name -->
                                        <div class="mb-3">
                                            <label class="form-label">Role Name</label>
                                            <input type="text" class="form-control" name="role_name" value="<?php echo htmlspecialchars($role['name']); ?>" required>
                                        </div>

                                        <!-- Permissions Checkboxes -->
                                        <div class="mb-3">
                                            <label class="form-label">Permissions</label>
                                            <div class="permission-grid p-3 border rounded">
                                                <?php
                                                $current_role_permissions = $role_permissions_map[$role['id']] ?? [];
                                                foreach ($all_permissions as $permission):
                                                    $is_checked = in_array($permission['id'], $current_role_permissions);
                                                ?>
                                                    <div class="form-check form-switch">
                                                        <input class="form-check-input" type="checkbox" name="permissions[]" value="<?php echo $permission['id']; ?>" id="perm-<?php echo $role['id']; ?>-<?php echo $permission['id']; ?>" <?php echo $is_checked ? 'checked' : ''; ?>>
                                                        <label class="form-check-label" for="perm-<?php echo $role['id']; ?>-<?php echo $permission['id']; ?>">
                                                            <?php echo htmlspecialchars($permission['name']); ?>
                                                        </label>
                                                        <small class="d-block text-muted"><?php echo htmlspecialchars($permission['description']); ?></small>
                                                    </div>
                                                <?php endforeach; ?>
                                            </div>
                                        </div>

                                        <!-- Action Buttons -->
                                        <div class="d-flex justify-content-between">
                                            <button type="submit" name="update_role" class="btn btn-success">
                                                <i class="fas fa-save me-2"></i>Update Role
                                            </button>
                                            
                                            <?php if (!in_array(strtolower($role['name']), ['admin', 'doctor', 'patient'])): ?>
                                            <button type="submit" name="delete_role" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete this role? This cannot be undone.');">
                                                <i class="fas fa-trash-alt me-2"></i>Delete Role
                                            </button>
                                            <?php endif; ?>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

```

### 2. Add a Link to the Admin Sidebar

To easily access this new page, add a link to it in the admin sidebar (`admin/includes/sidebar.php`).

```diff
--- a/c:/xampp/htdocs/fdms/admin/includes/sidebar.php
+++ b/c:/xampp/htdocs/fdms/admin/includes/sidebar.php