<?php
require_once '../config.php';
requireRole('admin');

$db = new Database();
$conn = $db->getConnection();

// --- Chart Data: New Patients by Month (Last 12 months) ---
$chart_data_sql = "
    SELECT DATE_FORMAT(u.created_at, '%Y-%m') as month, COUNT(p.id) as count
    FROM patients p
    JOIN users u ON p.user_id = u.id
    WHERE u.created_at >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
    GROUP BY month
    ORDER BY month ASC
";
$chart_stmt = $conn->query($chart_data_sql);
$monthly_registrations = $chart_stmt->fetchAll(PDO::FETCH_KEY_PAIR);

$chart_labels = [];
$chart_data = [];
for ($i = 11; $i >= 0; $i--) {
    $date = new DateTime("first day of -$i months");
    $month_key = $date->format('Y-m');
    $chart_labels[] = $date->format('M Y');
    $chart_data[] = $monthly_registrations[$month_key] ?? 0;
}

// --- Handle Doctor Assignment ---
$success_message = '';
$error_message = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['assign_doctor'])) {
    $patient_id = (int)$_POST['patient_id'];
    $doctor_id = !empty($_POST['doctor_id']) ? (int)$_POST['doctor_id'] : null;

    try {
        $stmt = $conn->prepare("UPDATE patients SET doctor_id = ? WHERE id = ?");
        $stmt->execute([$doctor_id, $patient_id]);
        $success_message = "Doctor assigned successfully.";
        logActivity($_SESSION['user_id'], "Assigned doctor ID: {$doctor_id} to patient ID: {$patient_id}");
    } catch (PDOException $e) {
        $error_message = "Database error: " . $e->getMessage();
    }
}

// --- Fetch all doctors for the assignment modal ---
$stmt_doctors = $conn->query("SELECT u.id, u.username FROM users u JOIN user_roles ur ON u.id = ur.user_id JOIN roles r ON ur.role_id = r.id WHERE r.name = 'doctor' ORDER BY u.username");
$all_doctors = $stmt_doctors->fetchAll();


// --- Filtering Logic ---
$filter_status = $_GET['status'] ?? 'all';
$search_term = sanitize($_GET['search'] ?? '');

$where_clauses = [];
$params = [];

if ($filter_status !== 'all' && $filter_status !== '') {
    if ($filter_status === 'inactive') {
        $where_clauses[] = "(p.subscription_status = 'inactive' OR p.subscription_status IS NULL)";
    } else {
        $where_clauses[] = "p.subscription_status = ?";
        $params[] = $filter_status;
    }
}

if ($search_term !== '') {
    $where_clauses[] = "(p.name LIKE ? OR u.email LIKE ? OR u.username LIKE ?)";
    $params[] = "%$search_term%";
    $params[] = "%$search_term%";
    $params[] = "%$search_term%";
}

$where_sql = '';
if (!empty($where_clauses)) {
    $where_sql = 'WHERE ' . implode(' AND ', $where_clauses);
}

// --- Pagination Logic ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 20;
$offset = ($page - 1) * $limit;

// Get total records for pagination
$count_stmt = $conn->prepare("SELECT COUNT(*) FROM patients p JOIN users u ON p.user_id = u.id $where_sql");
$count_stmt->execute($params);
$total_records = $count_stmt->fetchColumn();
$total_pages = ceil($total_records / $limit);

// --- Fetch Data ---
$sql = "
    SELECT p.id, p.name, u.email, u.username, u.status as user_status, p.subscription_status, p.subscription_end_date, u.created_at,
           p.doctor_id, doc.username as doctor_name
    FROM patients p
    JOIN users u ON p.user_id = u.id
    LEFT JOIN users doc ON p.doctor_id = doc.id
    $where_sql
    ORDER BY u.created_at DESC
    LIMIT $limit OFFSET $offset
";

$stmt = $conn->prepare($sql);
$stmt->execute($params);
$patients = $stmt->fetchAll();

function getStatusBadge($status) {
    if ($status === 'active') {
        return '<span class="badge bg-success">Active</span>';
    } elseif ($status === 'expired') {
        return '<span class="badge bg-danger">Expired</span>';
    } else {
        return '<span class="badge bg-secondary">Inactive</span>';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Patients - Admin - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: #f8f9fa;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
        }
        .card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0,0,0,0.12);
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-user-injured me-2"></i>Manage Patients</h1>
                </div>

                <?php if ($success_message): ?>
                    <div class="alert alert-success alert-dismissible fade show"><?php echo $success_message; ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div>
                <?php endif; ?>
                <?php if ($error_message): ?>
                    <div class="alert alert-danger alert-dismissible fade show"><?php echo $error_message; ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div>
                <?php endif; ?>

                <!-- Registration Chart -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="fas fa-chart-bar me-2"></i>New Patient Registrations (Last 12 Months)</h5>
                    </div>
                    <div class="card-body">
                        <canvas id="registrationChart"></canvas>
                    </div>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3 align-items-end">
                            <div class="col-md-5">
                                <label for="search" class="form-label">Search Patient</label>
                                <input type="text" class="form-control" id="search" name="search" placeholder="Name, Username, or Email" value="<?php echo htmlspecialchars($search_term); ?>">
                            </div>
                            <div class="col-md-4">
                                <label for="status" class="form-label">Filter by Subscription</label>
                                <select class="form-select" id="status" name="status">
                                    <option value="all" <?php if ($filter_status === 'all') echo 'selected'; ?>>All Statuses</option>
                                    <option value="active" <?php if ($filter_status === 'active') echo 'selected'; ?>>Active</option>
                                    <option value="expired" <?php if ($filter_status === 'expired') echo 'selected'; ?>>Expired</option>
                                    <option value="inactive" <?php if ($filter_status === 'inactive') echo 'selected'; ?>>Inactive</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <button type="submit" class="btn btn-primary w-100"><i class="fas fa-filter me-2"></i>Apply Filters</button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Patients Table -->
                <div class="card">
                    <div class="card-header">
                        <h5>Patient List (Showing <?php echo count($patients); ?> of <?php echo $total_records; ?>)</h5>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th>Name</th>
                                        <th>Username</th>
                                        <th>Email</th>
                                        <th>Subscription</th>
                                        <th>Sub. End Date</th>
                                        <th>Assigned Doctor</th>
                                        <th>Registered On</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($patients)): ?>
                                        <tr>
                                            <td colspan="8" class="text-center text-muted p-4">No patients found.</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($patients as $patient): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($patient['name']); ?></td>
                                                <td><?php echo htmlspecialchars($patient['username']); ?></td>
                                                <td><?php echo htmlspecialchars($patient['email']); ?></td>
                                                <td><?php echo getStatusBadge($patient['subscription_status']); ?></td>
                                                <td>
                                                    <?php if ($patient['subscription_end_date']): ?>
                                                        <?php echo date('M j, Y', strtotime($patient['subscription_end_date'])); ?>
                                                    <?php else: ?>
                                                        N/A
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if ($patient['doctor_name']): ?>
                                                        <span class="badge bg-info"><?php echo htmlspecialchars($patient['doctor_name']); ?></span>
                                                    <?php else: ?>
                                                        <span class="badge bg-secondary">Unassigned</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td><?php echo date('M j, Y', strtotime($patient['created_at'])); ?></td>
                                                <td>
                                                    <a href="users.php?search=<?php echo urlencode($patient['username']); ?>" class="btn btn-sm btn-outline-primary" title="Manage User">
                                                        <i class="fas fa-user-edit"></i>
                                                     </a>
                                                     <button class="btn btn-sm btn-outline-info" title="Assign Doctor" data-bs-toggle="modal" data-bs-target="#assignDoctorModal" data-patient-id="<?php echo $patient['id']; ?>" data-patient-name="<?php echo htmlspecialchars($patient['name']); ?>" data-doctor-id="<?php echo $patient['doctor_id']; ?>">
                                                        <i class="fas fa-user-md"></i>
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <?php if ($total_pages > 1): ?>
                    <div class="card-footer">
                        <!-- Pagination -->
                        <nav>
                            <ul class="pagination justify-content-center mb-0">
                                <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $page - 1; ?>&status=<?php echo $filter_status; ?>&search=<?php echo urlencode($search_term); ?>">Previous</a>
                                </li>
                                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                    <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $i; ?>&status=<?php echo $filter_status; ?>&search=<?php echo urlencode($search_term); ?>"><?php echo $i; ?></a>
                                    </li>
                                <?php endfor; ?>
                                <li class="page-item <?php echo $page >= $total_pages ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $page + 1; ?>&status=<?php echo $filter_status; ?>&search=<?php echo urlencode($search_term); ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>

    <!-- Assign Doctor Modal -->
    <div class="modal fade" id="assignDoctorModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Assign Doctor to <span id="modalPatientName"></span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="patient_id" id="modalPatientId">
                        <div class="mb-3">
                            <label for="doctor_id" class="form-label">Select Doctor</label>
                            <select class="form-select" id="modalDoctorId" name="doctor_id">
                                <option value="">-- Unassign --</option>
                                <?php foreach ($all_doctors as $doctor): ?>
                                    <option value="<?php echo $doctor['id']; ?>">
                                        <?php echo htmlspecialchars($doctor['username']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        <button type="submit" name="assign_doctor" class="btn btn-primary">Save Assignment</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const ctx = document.getElementById('registrationChart').getContext('2d');
            new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: <?php echo json_encode($chart_labels); ?>,
                    datasets: [{
                        label: 'New Patients',
                        data: <?php echo json_encode($chart_data); ?>,
                        backgroundColor: 'rgba(118, 75, 162, 0.6)',
                        borderColor: 'rgba(118, 75, 162, 1)',
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: { stepSize: 1 }
                        }
                    }
                }
            });
        });

        const assignDoctorModal = document.getElementById('assignDoctorModal');
        assignDoctorModal.addEventListener('show.bs.modal', function (event) {
            const button = event.relatedTarget;
            const patientId = button.getAttribute('data-patient-id');
            const patientName = button.getAttribute('data-patient-name');
            const currentDoctorId = button.getAttribute('data-doctor-id');

            const modalTitle = assignDoctorModal.querySelector('#modalPatientName');
            const modalPatientIdInput = assignDoctorM`odal.querySelector('#modalPatientId');
            const modalDoctorSelect = assignDoctorModal.querySelector('#modalDoctorId');

            modalTitle.textContent = patientName;
            modalPatientIdInput.value = patientId;
            modalDoctorSelect.value = currentDoctorId;
        });
    </script>
</body>
</html>