<?php
require_once '../config.php';
requireRole('admin');

$db = new Database();
$conn = $db->getConnection();

$success_message = '';
$error_message = '';

// --- Handle Approve/Reject Actions ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['approve_user'])) {
            $user_id = (int)$_POST['user_id'];

            // Update user status to active
            $stmt = $conn->prepare("UPDATE users SET status = 'active' WHERE id = ? AND status = 'pending_approval'");
            $stmt->execute([$user_id]);

            if ($stmt->rowCount() > 0) {
                // Notify the doctor
                createNotification($user_id, 'account_approved', 'Congratulations! Your account has been approved. You can now log in.', 'login.php', $_SESSION['user_id']);
                logActivity($_SESSION['user_id'], 'Approved doctor registration', "User ID: {$user_id}");
                $success_message = "Doctor account approved successfully.";
            } else {
                $error_message = "Could not approve account. It might have been already processed.";
            }
        }

        if (isset($_POST['reject_user'])) {
            $user_id = (int)$_POST['user_id'];

            // We need to delete from users, doctors, and user_roles tables
            $conn->beginTransaction();

            $stmt_del_doctor = $conn->prepare("DELETE FROM doctors WHERE user_id = ?");
            $stmt_del_doctor->execute([$user_id]);

            $stmt_del_roles = $conn->prepare("DELETE FROM user_roles WHERE user_id = ?");
            $stmt_del_roles->execute([$user_id]);

            $stmt_del_user = $conn->prepare("DELETE FROM users WHERE id = ? AND status = 'pending_approval'");
            $stmt_del_user->execute([$user_id]);

            if ($stmt_del_user->rowCount() > 0) {
                $conn->commit();
                logActivity($_SESSION['user_id'], 'Rejected doctor registration', "User ID: {$user_id}");
                $success_message = "Doctor registration rejected and user data deleted.";
            } else {
                $conn->rollBack();
                $error_message = "Could not reject account. It might have been already processed.";
            }
        }
    } catch (PDOException $e) {
        if ($conn->inTransaction()) {
            $conn->rollBack();
        }
        $error_message = "Database error: " . $e->getMessage();
    }
}

// --- Fetch Pending Doctors ---
$stmt = $conn->prepare("
    SELECT u.id, u.username, u.email, u.created_at, d.name as full_name, d.specialization, d.license_number
    FROM users u
    JOIN doctors d ON u.id = d.user_id
    WHERE u.status = 'pending_approval' AND u.role = 'doctor'
    ORDER BY u.created_at ASC
");
$stmt->execute();
$pending_doctors = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pending Approvals - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-user-clock me-2"></i>Pending Doctor Approvals</h1>
                </div>

                <?php if ($success_message): ?><div class="alert alert-success"><?php echo $success_message; ?></div><?php endif; ?>
                <?php if ($error_message): ?><div class="alert alert-danger"><?php echo $error_message; ?></div><?php endif; ?>

                <div class="card">
                    <div class="card-header">
                        <h5>Doctors Awaiting Review (<?php echo count($pending_doctors); ?>)</h5>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th>Full Name</th>
                                        <th>Email</th>
                                        <th>Specialization</th>
                                        <th>License No.</th>
                                        <th>Registered On</th>
                                        <th class="text-end">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($pending_doctors)): ?>
                                        <tr><td colspan="6" class="text-center text-muted p-4">No pending approvals.</td></tr>
                                    <?php else: ?>
                                        <?php foreach ($pending_doctors as $doctor): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($doctor['full_name']); ?></td>
                                                <td><?php echo htmlspecialchars($doctor['email']); ?></td>
                                                <td><?php echo htmlspecialchars($doctor['specialization']); ?></td>
                                                <td><?php echo htmlspecialchars($doctor['license_number']); ?></td>
                                                <td><?php echo date('M j, Y H:i', strtotime($doctor['created_at'])); ?></td>
                                                <td class="text-end">
                                                    <form method="POST" class="d-inline">
                                                        <input type="hidden" name="user_id" value="<?php echo $doctor['id']; ?>">
                                                        <button type="submit" name="approve_user" class="btn btn-sm btn-success"><i class="fas fa-check me-1"></i> Approve</button>
                                                        <button type="submit" name="reject_user" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to reject and delete this user?');"><i class="fas fa-times me-1"></i> Reject</button>
                                                    </form>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>