<?php
require_once '../config.php';
requireRole('admin');

$db = new Database();
$conn = $db->getConnection();

// --- Filtering and Search Logic ---
$search_term = sanitize($_GET['search'] ?? '');
$risk_filter = sanitize($_GET['risk'] ?? '');

$where_clauses = [];
$params = [];

if ($search_term !== '') {
    $where_clauses[] = "(p.name LIKE ? OR u.username LIKE ?)";
    $params[] = "%$search_term%";
    $params[] = "%$search_term%";
}

if ($risk_filter !== '') {
    $where_clauses[] = "a.risk_level = ?";
    $params[] = $risk_filter;
}

$where_sql = !empty($where_clauses) ? 'WHERE ' . implode(' AND ', $where_clauses) : '';

// --- Pagination Logic ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 20;
$offset = ($page - 1) * $limit;

// Get total records for pagination
$count_sql = "
    SELECT COUNT(a.id) 
    FROM pcos_comprehensive_assessments a
    JOIN patients p ON a.patient_id = p.id
    JOIN users u ON p.user_id = u.id
    $where_sql
";
$count_stmt = $conn->prepare($count_sql);
$count_stmt->execute($params);
$total_records = $count_stmt->fetchColumn();
$total_pages = ceil($total_records / $limit);

// --- Fetch Data ---
$sql = "
    SELECT a.*, p.name as patient_name, u.username
    FROM pcos_comprehensive_assessments a
    JOIN patients p ON a.patient_id = p.id
    JOIN users u ON p.user_id = u.id
    $where_sql
    ORDER BY a.created_at DESC
    LIMIT $limit OFFSET $offset
";

$stmt = $conn->prepare($sql);
$stmt->execute($params);
$assessments = $stmt->fetchAll();

function getRiskLevelClass($risk_level) {
    switch (strtolower($risk_level)) {
        case 'high': return 'danger';
        case 'moderate-high': return 'warning';
        case 'moderate': return 'info';
        case 'low': return 'success';
        default: return 'secondary';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Medical Records - Admin - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: #c0bbf5ff;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
        }
        .card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0,0,0,0.12);
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-file-medical-alt me-2"></i>All Medical Records</h1>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3 align-items-end">
                            <div class="col-md-5">
                                <label for="search" class="form-label">Search Patient</label>
                                <input type="text" class="form-control" id="search" name="search" placeholder="Name or Username" value="<?php echo htmlspecialchars($search_term); ?>">
                            </div>
                            <div class="col-md-4">
                                <label for="risk" class="form-label">Filter by Risk Level</label>
                                <select class="form-select" id="risk" name="risk">
                                    <option value="">All Risk Levels</option>
                                    <option value="High" <?php if ($risk_filter === 'High') echo 'selected'; ?>>High</option>
                                    <option value="Moderate-High" <?php if ($risk_filter === 'Moderate-High') echo 'selected'; ?>>Moderate-High</option>
                                    <option value="Moderate" <?php if ($risk_filter === 'Moderate') echo 'selected'; ?>>Moderate</option>
                                    <option value="Low" <?php if ($risk_filter === 'Low') echo 'selected'; ?>>Low</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <button type="submit" class="btn btn-primary w-100"><i class="fas fa-filter me-2"></i>Apply Filters</button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Records Table -->
                <div class="card">
                    <div class="card-header">
                        <h5>Assessment Records (Showing <?php echo count($assessments); ?> of <?php echo $total_records; ?>)</h5>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th>Patient</th>
                                        <th>Assessment Date</th>
                                        <th>PCOS Probability</th>
                                        <th>Risk Level</th>
                                        <th>Diagnosis Category</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($assessments)): ?>
                                        <tr>
                                            <td colspan="6" class="text-center text-muted p-4">No records found.</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($assessments as $record): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($record['patient_name']); ?> <small class="text-muted">(<?php echo htmlspecialchars($record['username']); ?>)</small></td>
                                                <td><?php echo date('M j, Y', strtotime($record['created_at'])); ?></td>
                                                <td><span class="badge bg-<?php echo $record['pcos_probability'] >= 70 ? 'danger' : ($record['pcos_probability'] >= 40 ? 'warning' : 'success'); ?>"><?php echo $record['pcos_probability']; ?>%</span></td>
                                                <td><span class="badge bg-<?php echo getRiskLevelClass($record['risk_level']); ?>"><?php echo htmlspecialchars($record['risk_level']); ?></span></td>
                                                <td><?php echo htmlspecialchars($record['diagnosis_category']); ?></td>
                                                <td>
                                                    <button class="btn btn-sm btn-info" data-bs-toggle="modal" data-bs-target="#detailsModal" data-assessment-id="<?php echo $record['id']; ?>">
                                                        <i class="fas fa-eye me-1"></i> View Details
                                                    </button>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <?php if ($total_pages > 1): ?>
                    <div class="card-footer">
                        <!-- Pagination -->
                        <nav>
                            <ul class="pagination justify-content-center mb-0">
                                <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $page - 1; ?>&risk=<?php echo $risk_filter; ?>&search=<?php echo urlencode($search_term); ?>">Previous</a>
                                </li>
                                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                    <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $i; ?>&risk=<?php echo $risk_filter; ?>&search=<?php echo urlencode($search_term); ?>"><?php echo $i; ?></a>
                                    </li>
                                <?php endfor; ?>
                                <li class="page-item <?php echo $page >= $total_pages ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $page + 1; ?>&risk=<?php echo $risk_filter; ?>&search=<?php echo urlencode($search_term); ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>

    <!-- Details Modal -->
    <div class="modal fade" id="detailsModal" tabindex="-1" aria-labelledby="detailsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-dialog-scrollable">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="detailsModalLabel">Assessment Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" id="modalBodyContent">
                    <!-- Content will be loaded here by JavaScript -->
                    <div class="text-center">
                        <div class="spinner-border text-primary" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function () {
        const detailsModal = document.getElementById('detailsModal');
        const assessmentsData = <?php echo json_encode($assessments); ?>;
        
        // Create a map for quick lookup
        const assessmentMap = new Map(assessmentsData.map(assessment => [assessment.id.toString(), assessment]));

        detailsModal.addEventListener('show.bs.modal', function (event) {
            const button = event.relatedTarget;
            const assessmentId = button.getAttribute('data-assessment-id');
            const assessment = assessmentMap.get(assessmentId);
            
            const modalBody = document.getElementById('modalBodyContent');
            
            if (assessment) {
                let content = `
                    <h5>Patient: ${escapeHtml(assessment.patient_name)} (${escapeHtml(assessment.username)})</h5>
                    <p class="text-muted">Assessment Date: ${new Date(assessment.created_at).toLocaleString()}</p>
                    <hr>
                    <div class="row">
                        <div class="col-md-4"><strong>PCOS Probability:</strong> <span class="badge bg-primary">${assessment.pcos_probability}%</span></div>
                        <div class="col-md-4"><strong>Risk Level:</strong> <span class="badge bg-info">${escapeHtml(assessment.risk_level)}</span></div>
                        <div class="col-md-4"><strong>Diagnosis:</strong> ${escapeHtml(assessment.diagnosis_category)}</div>
                    </div>
                    <hr>
                    <h6>Key Metrics:</h6>
                    <div class="row">
                `;

                // Create a list of key-value pairs from the assessment object
                const detailsToShow = {
                    'BMI': assessment.bmi,
                    'Waist-Hip Ratio': assessment.waist_hip_ratio,
                    'Hirsutism Score': assessment.hirsutism_score,
                    'Cycle Length': assessment.cycle_length,
                    'LH/FSH Ratio': assessment.lh_fsh_ratio,
                    'Total Testosterone': assessment.total_testosterone,
                    'Follicle Count (L)': assessment.follicle_count_left,
                    'Follicle Count (R)': assessment.follicle_count_right,
                    'Acne Severity': assessment.acne_severity,
                    'Insulin Resistance': assessment.insulin_resistance,
                    'Family History (PCOS)': assessment.family_history_pcos,
                    'Sleep Apnea': assessment.sleep_apnea
                };

                for (const [key, value] of Object.entries(detailsToShow)) {
                    content += `
                        <div class="col-md-6 col-lg-4 mb-2">
                            <strong>${key}:</strong> ${escapeHtml(value !== null ? value : 'N/A')}
                        </div>
                    `;
                }

                content += `</div>`;
                modalBody.innerHTML = content;
            } else {
                modalBody.innerHTML = '<p class="text-danger">Could not find assessment details.</p>';
            }
        });

        function escapeHtml(unsafe) {
            if (unsafe === null || typeof unsafe === 'undefined') {
                return '';
            }
            return unsafe
                 .toString()
                 .replace(/&/g, "&amp;")
                 .replace(/</g, "&lt;")
                 .replace(/>/g, "&gt;")
                 .replace(/"/g, "&quot;")
                 .replace(/'/g, "&#039;");
        }
    });
    </script>
</body>
</html>