<?php
require_once '../config.php';
requireRole('admin'); // Ensure only admins can access

$db = new Database();
$conn = $db->getConnection();

// --- Filtering Logic ---
$filter_status = $_GET['status'] ?? 'all';
$search_term = $_GET['search'] ?? '';

$where_clauses = [];
$params = [];

if ($filter_status !== 'all' && $filter_status !== '') {
    // Handle 'inactive' which might be NULL or 'inactive'
    if ($filter_status === 'inactive') {
        $where_clauses[] = "(p.subscription_status = 'inactive' OR p.subscription_status IS NULL)";
    } else {
        $where_clauses[] = "p.subscription_status = ?";
        $params[] = $filter_status;
    }
}

if ($search_term !== '') {
    $where_clauses[] = "(p.name LIKE ? OR u.email LIKE ?)";
    $params[] = "%$search_term%";
    $params[] = "%$search_term%";
}

$where_sql = '';
if (!empty($where_clauses)) {
    $where_sql = 'WHERE ' . implode(' AND ', $where_clauses);
}

// --- Pagination Logic ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 20;
$offset = ($page - 1) * $limit;

// Get total records for pagination
$count_stmt = $conn->prepare("SELECT COUNT(*) FROM patients p JOIN users u ON p.user_id = u.id $where_sql");
$count_stmt->execute($params);
$total_records = $count_stmt->fetchColumn();
$total_pages = ceil($total_records / $limit);

// --- Fetch Data ---
$sql = "
    SELECT p.id, p.name, u.email, p.subscription_status, p.subscription_end_date
    FROM patients p
    JOIN users u ON p.user_id = u.id
    $where_sql
    ORDER BY p.subscription_end_date DESC
    LIMIT $limit OFFSET $offset
";

$stmt = $conn->prepare($sql);
$stmt->execute($params);
$subscriptions = $stmt->fetchAll();

// --- Stats ---
$stats_stmt = $conn->query("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN subscription_status = 'active' THEN 1 ELSE 0 END) as active,
        SUM(CASE WHEN subscription_status = 'expired' THEN 1 ELSE 0 END) as expired,
        SUM(CASE WHEN subscription_status = 'inactive' OR subscription_status IS NULL THEN 1 ELSE 0 END) as inactive
    FROM patients
");
$stats = $stats_stmt->fetch();

function getStatusBadge($status) {
    if ($status === 'active') {
        return '<span class="badge bg-success">Active</span>';
    } elseif ($status === 'expired') {
        return '<span class="badge bg-danger">Expired</span>';
    } else {
        return '<span class="badge bg-secondary">Inactive</span>';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Subscription Report - Admin - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: #f8f9fa;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
        }
        .card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0,0,0,0.12);
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4" style="background-color: #f8f9fa;">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-chart-bar me-2"></i>Subscription Report</h1>
                </div>

                <!-- Stats Cards -->
                <div class="row mb-4">
                    <div class="col-md-3 mb-3">
                        <div class="card text-white bg-primary h-100">
                            <div class="card-body">
                                <h5 class="card-title"><i class="fas fa-users me-2"></i>Total Patients</h5>
                                <p class="card-text fs-4 fw-bold"><?php echo $stats['total'] ?? 0; ?></p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-3">
                        <div class="card text-white bg-success h-100">
                            <div class="card-body">
                                <h5 class="card-title"><i class="fas fa-check-circle me-2"></i>Active Subscriptions</h5>
                                <p class="card-text fs-4 fw-bold"><?php echo $stats['active'] ?? 0; ?></p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-3">
                        <div class="card text-white bg-danger h-100">
                            <div class="card-body">
                                <h5 class="card-title"><i class="fas fa-times-circle me-2"></i>Expired Subscriptions</h5>
                                <p class="card-text fs-4 fw-bold"><?php echo $stats['expired'] ?? 0; ?></p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-3">
                        <div class="card text-white bg-secondary h-100">
                            <div class="card-body">
                                <h5 class="card-title"><i class="fas fa-minus-circle me-2"></i>Inactive Patients</h5>
                                <p class="card-text fs-4 fw-bold"><?php echo $stats['inactive'] ?? 0; ?></p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3 align-items-end">
                            <div class="col-md-5">
                                <label for="search" class="form-label">Search Patient</label>
                                <input type="text" class="form-control" id="search" name="search" placeholder="Name or Email" value="<?php echo htmlspecialchars($search_term); ?>">
                            </div>
                            <div class="col-md-4">
                                <label for="status" class="form-label">Filter by Status</label>
                                <select class="form-select" id="status" name="status">
                                    <option value="all" <?php if ($filter_status === 'all') echo 'selected'; ?>>All Statuses</option>
                                    <option value="active" <?php if ($filter_status === 'active') echo 'selected'; ?>>Active</option>
                                    <option value="expired" <?php if ($filter_status === 'expired') echo 'selected'; ?>>Expired</option>
                                    <option value="inactive" <?php if ($filter_status === 'inactive') echo 'selected'; ?>>Inactive</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <button type="submit" class="btn btn-primary w-100"><i class="fas fa-filter me-2"></i>Apply</button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Subscription Table -->
                <div class="card">
                    <div class="card-header">
                        <h5>Subscription Details (Showing <?php echo count($subscriptions); ?> of <?php echo $total_records; ?>)</h5>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th>Patient Name</th>
                                        <th>Email</th>
                                        <th>Status</th>
                                        <th>Subscription End Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($subscriptions)): ?>
                                        <tr>
                                            <td colspan="4" class="text-center text-muted p-4">No subscriptions found for the selected filters.</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($subscriptions as $sub): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($sub['name']); ?></td>
                                                <td><?php echo htmlspecialchars($sub['email']); ?></td>
                                                <td><?php echo getStatusBadge($sub['subscription_status']); ?></td>
                                                <td>
                                                    <?php if ($sub['subscription_end_date']): ?>
                                                        <?php echo date('F j, Y', strtotime($sub['subscription_end_date'])); ?>
                                                        <?php if ($sub['subscription_status'] === 'active' && new DateTime() < new DateTime($sub['subscription_end_date']) && new DateTime() > (new DateTime($sub['subscription_end_date']))->sub(new DateInterval('P3D'))): ?>
                                                            <span class="badge bg-warning text-dark ms-2">Expiring Soon</span>
                                                        <?php endif; ?>
                                                    <?php else: ?>
                                                        N/A
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <?php if ($total_pages > 1): ?>
                    <div class="card-footer">
                        <!-- Pagination -->
                        <nav>
                            <ul class="pagination justify-content-center mb-0">
                                <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $page - 1; ?>&status=<?php echo $filter_status; ?>&search=<?php echo urlencode($search_term); ?>">Previous</a>
                                </li>
                                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                    <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $i; ?>&status=<?php echo $filter_status; ?>&search=<?php echo urlencode($search_term); ?>"><?php echo $i; ?></a>
                                    </li>
                                <?php endfor; ?>
                                <li class="page-item <?php echo $page >= $total_pages ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $page + 1; ?>&status=<?php echo $filter_status; ?>&search=<?php echo urlencode($search_term); ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>