<?php
require_once '../config.php';
requireRole('admin');

$db = new Database();
$conn = $db->getConnection();

$success_message = '';
$error_message = '';

// --- Handle Form Submissions ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // --- Handle Role Update ---
        if (isset($_POST['update_user_roles'])) {
            $user_id = (int)$_POST['user_id'];
            $role_ids = $_POST['roles'] ?? [];

            $conn->beginTransaction();

            // 1. Delete existing roles for the user
            $stmt_delete = $conn->prepare("DELETE FROM user_roles WHERE user_id = ?");
            $stmt_delete->execute([$user_id]);

            // 2. Insert new roles
            if (!empty($role_ids)) {
                $stmt_insert = $conn->prepare("INSERT INTO user_roles (user_id, role_id) VALUES (?, ?)");
                foreach ($role_ids as $role_id) {
                    $stmt_insert->execute([$user_id, (int)$role_id]);
                }
            }

            $conn->commit();
            $success_message = "User roles updated successfully.";
            logActivity($_SESSION['user_id'], "Updated roles for user ID: {$user_id}");
        }

        // --- Handle Status Toggle ---
        if (isset($_POST['toggle_status'])) {
            $user_id = (int)$_POST['user_id'];
            $current_status = $_POST['current_status'];
            $new_status = $current_status === 'active' ? 'inactive' : 'active';

            $stmt = $conn->prepare("UPDATE users SET status = ? WHERE id = ?");
            $stmt->execute([$new_status, $user_id]);
            $success_message = "User status updated to '{$new_status}'.";
            logActivity($_SESSION['user_id'], "Updated user status to {$new_status} for user ID: {$user_id}");
        }

        // --- Handle User Deletion ---
        if (isset($_POST['delete_user'])) {
            $user_id = (int)$_POST['user_id'];
            $stmt = $conn->prepare("DELETE FROM users WHERE id = ? AND id != ?");
            $stmt->execute([$user_id, $_SESSION['user_id']]);
            $success_message = "User deleted successfully.";
            logActivity($_SESSION['user_id'], "Deleted user ID: {$user_id}");
        }

    } catch (PDOException $e) {
        if ($conn->inTransaction()) {
            $conn->rollBack();
        }
        $error_message = "Database error: " . $e->getMessage();
    }
}

// --- Fetch Data for Display ---

// Filters and Pagination
$search = sanitize($_GET['search'] ?? '');
$role_filter = (int)($_GET['role'] ?? 0);
$page = (int)($_GET['page'] ?? 1);
$limit = 15;
$offset = ($page - 1) * $limit;

$where_conditions = [];
$params = [];

if ($search) {
    $where_conditions[] = '(u.username LIKE ? OR u.email LIKE ?)';
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if ($role_filter > 0) {
    $where_conditions[] = 'ur.role_id = ?';
    $params[] = $role_filter;
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

$count_sql = "SELECT COUNT(DISTINCT u.id) FROM users u ";
if ($role_filter > 0) {
    $count_sql .= "LEFT JOIN user_roles ur ON u.id = ur.user_id ";
}
$count_sql .= $where_clause;

$stmt_count = $conn->prepare($count_sql);
$stmt_count->execute($params);
$total_users = $stmt_count->fetchColumn();
$total_pages = ceil($total_users / $limit);

$sql = "SELECT u.id, u.username, u.email, u.status, u.created_at, GROUP_CONCAT(r.name SEPARATOR ', ') as roles
        FROM users u
        LEFT JOIN user_roles ur ON u.id = ur.user_id
        LEFT JOIN roles r ON ur.role_id = r.id
        $where_clause
        GROUP BY u.id
        ORDER BY u.created_at DESC
        LIMIT $limit OFFSET $offset";

$stmt_users = $conn->prepare($sql);
$stmt_users->execute($params);
$users = $stmt_users->fetchAll();

// Fetch all available roles for the filter and modal
$all_roles = $conn->query("SELECT id, name FROM roles ORDER BY name")->fetchAll();

// Fetch all user-role mappings for the modal
$user_roles_map = [];
$stmt_user_roles = $conn->query("SELECT user_id, role_id FROM user_roles");
while ($row = $stmt_user_roles->fetch()) {
    $user_roles_map[$row['user_id']][] = $row['role_id'];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Users - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;4_00;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: #e1b1e8ff;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
        }
        .card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0,0,0,0.12);
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>

            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-users-cog me-2"></i>Manage Users</h1>
                </div>

                <?php if ($success_message): ?>
                    <div class="alert alert-success alert-dismissible fade show"><?php echo $success_message; ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div>
                <?php endif; ?>
                <?php if ($error_message): ?>
                    <div class="alert alert-danger alert-dismissible fade show"><?php echo $error_message; ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div>
                <?php endif; ?>

                <!-- User Table -->
                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Username</th>
                                        <th>Email</th>
                                        <th>Roles</th>
                                        <th>Status</th>
                                        <th>Registered On</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($users as $user): ?>
                                    <tr>
                                        <td><?php echo $user['id']; ?></td>
                                        <td><?php echo htmlspecialchars($user['username']); ?></td>
                                        <td><?php echo htmlspecialchars($user['email']); ?></td>
                                        <td>
                                            <?php if ($user['roles']): ?>
                                                <?php foreach(explode(', ', $user['roles']) as $role): ?>
                                                    <span class="badge bg-primary"><?php echo htmlspecialchars($role); ?></span>
                                                <?php endforeach; ?>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">No Role</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php echo $user['status'] == 'active' ? 'success' : 'secondary'; ?>">
                                                <?php echo ucfirst($user['status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo date('M j, Y', strtotime($user['created_at'])); ?></td>
                                        <td>
                                            <?php if ($user['id'] != $_SESSION['user_id']): ?>
                                                <button class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#editRolesModal" data-user-id="<?php echo $user['id']; ?>" data-username="<?php echo htmlspecialchars($user['username']); ?>">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                    <input type="hidden" name="current_status" value="<?php echo $user['status']; ?>">
                                                    <button type="submit" name="toggle_status" class="btn btn-sm btn-<?php echo $user['status'] == 'active' ? 'warning' : 'success'; ?>" title="<?php echo $user['status'] == 'active' ? 'Deactivate' : 'Activate'; ?>">
                                                        <i class="fas fa-<?php echo $user['status'] == 'active' ? 'user-slash' : 'user-check'; ?>"></i>
                                                    </button>
                                                </form>
                                                <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this user? This cannot be undone.');">
                                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                    <button type="submit" name="delete_user" class="btn btn-sm btn-danger" title="Delete User">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            <?php else: ?>
                                                <span class="text-muted">Current User</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

            </main>
        </div>
    </div>

    <!-- Edit Roles Modal -->
    <div class="modal fade" id="editRolesModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Roles for <span id="modalUsername"></span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="user_id" id="modalUserId">
                        <p>Select the roles for this user:</p>
                        <div id="modalRolesCheckboxes">
                            <!-- Checkboxes will be populated by JavaScript -->
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        <button type="submit" name="update_user_roles" class="btn btn-primary">Save Changes</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function () {
        const editRolesModal = document.getElementById('editRolesModal');
        const allRoles = <?php echo json_encode($all_roles); ?>;
        const userRolesMap = <?php echo json_encode($user_roles_map); ?>;

        editRolesModal.addEventListener('show.bs.modal', function (event) {
            const button = event.relatedTarget;
            const userId = button.getAttribute('data-user-id');
            const username = button.getAttribute('data-username');

            document.getElementById('modalUsername').textContent = username;
            document.getElementById('modalUserId').value = userId;

            const checkboxesContainer = document.getElementById('modalRolesCheckboxes');
            checkboxesContainer.innerHTML = ''; // Clear previous checkboxes

            const currentUserRoles = userRolesMap[userId] || [];

            allRoles.forEach(role => {
                const isChecked = currentUserRoles.includes(parseInt(role.id));
                const checkboxHtml = `
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="roles[]" value="${role.id}" id="role-${role.id}" ${isChecked ? 'checked' : ''}>
                        <label class="form-check-label" for="role-${role.id}">
                            ${role.name}
                        </label>
                    </div>
                `;
                checkboxesContainer.insertAdjacentHTML('beforeend', checkboxHtml);
            });
        });
    });
    </script>
</body>
</html>