<?php
require_once '../config.php';
requireRole('doctor');

$db = new Database();
$conn = $db->getConnection();

// Get doctor info
$stmt = $conn->prepare("SELECT * FROM doctors WHERE user_id = ?");
$stmt->execute([$_SESSION['user_id']]);
$doctor = $stmt->fetch();

$success = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $patient_id = (int)$_POST['patient_id'];
    $examination_date = $_POST['examination_date'];
    
    // Ultrasound results
    $follicle_count_right = $_POST['follicle_count_right'] ? (int)$_POST['follicle_count_right'] : null;
    $follicle_count_left = $_POST['follicle_count_left'] ? (int)$_POST['follicle_count_left'] : null;
    $follicle_size_right = sanitize($_POST['follicle_size_right']);
    $follicle_size_left = sanitize($_POST['follicle_size_left']);
    $endometrial_thickness = $_POST['endometrial_thickness'] ? (float)$_POST['endometrial_thickness'] : null;
    $ovary_volume_right = $_POST['ovary_volume_right'] ? (float)$_POST['ovary_volume_right'] : null;
    $ovary_volume_left = $_POST['ovary_volume_left'] ? (float)$_POST['ovary_volume_left'] : null;
    $dominant_follicle_right = isset($_POST['dominant_follicle_right']) ? 1 : 0;
    $dominant_follicle_left = isset($_POST['dominant_follicle_left']) ? 1 : 0;
    $dominant_follicle_size = $_POST['dominant_follicle_size'] ? (float)$_POST['dominant_follicle_size'] : null;
    
    // Hormonal levels
    $fsh_level = $_POST['fsh_level'] ? (float)$_POST['fsh_level'] : null;
    $lh_level = $_POST['lh_level'] ? (float)$_POST['lh_level'] : null;
    $amh_level = $_POST['amh_level'] ? (float)$_POST['amh_level'] : null;
    $estradiol_level = $_POST['estradiol_level'] ? (float)$_POST['estradiol_level'] : null;
    $progesterone_level = $_POST['progesterone_level'] ? (float)$_POST['progesterone_level'] : null;
    $testosterone_level = $_POST['testosterone_level'] ? (float)$_POST['testosterone_level'] : null;
    $tsh_level = $_POST['tsh_level'] ? (float)$_POST['tsh_level'] : null;
    $prolactin_level = $_POST['prolactin_level'] ? (float)$_POST['prolactin_level'] : null;
    
    // Metabolic markers
    $fasting_glucose = $_POST['fasting_glucose'] ? (float)$_POST['fasting_glucose'] : null;
    $fasting_insulin = $_POST['fasting_insulin'] ? (float)$_POST['fasting_insulin'] : null;
    $hba1c = $_POST['hba1c'] ? (float)$_POST['hba1c'] : null;
    
    $diagnosis = sanitize($_POST['diagnosis']);
    $treatment_plan = sanitize($_POST['treatment_plan']);
    $notes = sanitize($_POST['notes']);
    
    // PCOS symptoms
    $irregular_cycles = sanitize($_POST['irregular_cycles']);
    $hirsutism = sanitize($_POST['hirsutism']);
    $acne = sanitize($_POST['acne']);
    $hair_loss = sanitize($_POST['hair_loss']);
    $weight_issues = sanitize($_POST['weight_issues']);
    $skin_tags = sanitize($_POST['skin_tags']);
    $dark_skin_patches = sanitize($_POST['dark_skin_patches']);
    
    if (empty($patient_id) || empty($examination_date)) {
        $error = 'Please select a patient and examination date.';
    } else {
        try {
            $conn->beginTransaction();
            
            // Calculate PCOS score
            $pcosData = [
                'irregular_cycles' => $irregular_cycles,
                'hirsutism' => $hirsutism,
                'acne' => $acne,
                'hair_loss' => $hair_loss,
                'weight_issues' => $weight_issues,
                'amh_level' => $amh_level,
                'lh_level' => $lh_level,
                'fsh_level' => $fsh_level
            ];
            
            // Calculate PCOS score based on symptoms and hormone levels
            $pcos_score = 0;
            
            // Add points for clinical symptoms
            if ($pcosData['irregular_cycles'] === 'yes') $pcos_score += 2;
            if ($pcosData['hirsutism'] === 'yes') $pcos_score += 2;
            if ($pcosData['acne'] === 'yes') $pcos_score += 1;
            if ($pcosData['hair_loss'] === 'yes') $pcos_score += 1;
            if ($pcosData['weight_issues'] === 'yes') $pcos_score += 1;
            
            // Add points for hormone levels
            if ($pcosData['amh_level'] > 4.0) $pcos_score += 2;
            if ($pcosData['lh_level'] > 10.0) $pcos_score += 2;
            if ($pcosData['lh_level'] && $pcosData['fsh_level'] && 
                ($pcosData['lh_level'] / $pcosData['fsh_level']) > 2) {
                $pcos_score += 2;
            }
            // Determine PCOS risk level based on score
            $pcos_risk_level = 'Low';
            if ($pcos_score >= 8) {
                $pcos_risk_level = 'High';
            } elseif ($pcos_score >= 4) {
                $pcos_risk_level = 'Moderate';
            }
            
            // Insert follicle record
            $stmt = $conn->prepare("INSERT INTO follicle_records 
                (patient_id, doctor_id, examination_date, follicle_count_right, follicle_count_left, 
                follicle_size_right, follicle_size_left, endometrial_thickness, ovary_volume_right, 
                ovary_volume_left, dominant_follicle_right, dominant_follicle_left, dominant_follicle_size,
                fsh_level, lh_level, amh_level, estradiol_level, progesterone_level, testosterone_level,
                tsh_level, prolactin_level, fasting_glucose, fasting_insulin, hba1c, diagnosis, 
                treatment_plan, notes, pcos_score, pcos_risk_level) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            
            $stmt->execute([
                $patient_id, $doctor['id'], $examination_date, $follicle_count_right, $follicle_count_left,
                $follicle_size_right, $follicle_size_left, $endometrial_thickness, $ovary_volume_right,
                $ovary_volume_left, $dominant_follicle_right, $dominant_follicle_left, $dominant_follicle_size,
                $fsh_level, $lh_level, $amh_level, $estradiol_level, $progesterone_level, $testosterone_level,
                $tsh_level, $prolactin_level, $fasting_glucose, $fasting_insulin, $hba1c, $diagnosis,
                $treatment_plan, $notes, $pcos_score, $pcos_risk_level
            ]);
            
            $record_id = $conn->lastInsertId();
            
            // Insert PCOS symptoms
            $stmt = $conn->prepare("INSERT INTO pcos_symptoms 
                (patient_id, record_id, irregular_cycles, hirsutism, acne, hair_loss, weight_issues, skin_tags, dark_skin_patches) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            
            $stmt->execute([
                $patient_id, $record_id, $irregular_cycles, $hirsutism, $acne, $hair_loss, $weight_issues, $skin_tags, $dark_skin_patches
            ]);
            
            $conn->commit();
            
            $success = 'Record added successfully! PCOS Score: ' . $pcos_score . ' (' . $pcos_risk_level . ' risk)';
            logActivity($_SESSION['user_id'], 'Added follicle record', "Patient ID: $patient_id, PCOS Score: $pcos_score");
        } catch (Exception $e) {
            $conn->rollBack();
            $error = 'Failed to add record: ' . $e->getMessage();
        }
    }
}

// Get all patients
$stmt = $conn->query("SELECT * FROM patients ORDER BY name");
$patients = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Record - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .section-header {
            background-color: #f8f9fa;
            padding: 10px 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            border-left: 4px solid #0d6efd;
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Add Follicle Record</h1>
                    <a href="records.php" class="btn btn-secondary">
                        <i class="fas fa-list me-2"></i>View All Records
                    </a>
                </div>
                
                <?php if ($success): ?>
                    <div class="alert alert-success alert-dismissible fade show">
                        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">New Follicle Diagnostic Record</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="patient_id" class="form-label">Patient *</label>
                                    <select class="form-select" id="patient_id" name="patient_id" required>
                                        <option value="">Select Patient</option>
                                        <?php foreach ($patients as $patient): ?>
                                            <option value="<?php echo $patient['id']; ?>">
                                                <?php echo htmlspecialchars($patient['name']) . ' (Age: ' . $patient['age'] . ')'; ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="examination_date" class="form-label">Examination Date *</label>
                                    <input type="date" class="form-control" id="examination_date" name="examination_date" required max="<?php echo date('Y-m-d'); ?>">
                                </div>
                            </div>
                            
                            <div class="section-header">
                                <h6 class="mb-0">Ultrasound Results</h6>
                            </div>
                            <div class="row">
                                <div class="col-md-6">
                                    <h6>Right Ovary</h6>
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="follicle_count_right" class="form-label">Follicle Count</label>
                                            <input type="number" class="form-control" id="follicle_count_right" name="follicle_count_right" min="0" placeholder="e.g., 8-12 (normal range)">
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="follicle_size_right" class="form-label">Follicle Size (mm)</label>
                                            <input type="text" class="form-control" id="follicle_size_right" name="follicle_size_right" placeholder="e.g., 5-8mm">
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="ovary_volume_right" class="form-label">Ovary Volume (cm³)</label>
                                            <input type="number" step="0.01" class="form-control" id="ovary_volume_right" name="ovary_volume_right" placeholder="e.g., 3-8 cm³ (normal range)">
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <div class="form-check mt-4">
                                                <input class="form-check-input" type="checkbox" id="dominant_follicle_right" name="dominant_follicle_right">
                                                <label class="form-check-label" for="dominant_follicle_right">Dominant Follicle</label>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <h6>Left Ovary</h6>
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="follicle_count_left" class="form-label">Follicle Count</label>
                                            <input type="number" class="form-control" id="follicle_count_left" name="follicle_count_left" min="0" placeholder="e.g., 8-12 (normal range)">
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="follicle_size_left" class="form-label">Follicle Size (mm)</label>
                                            <input type="text" class="form-control" id="follicle_size_left" name="follicle_size_left" placeholder="e.g., 6-9mm">
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="ovary_volume_left" class="form-label">Ovary Volume (cm³)</label>
                                            <input type="number" step="0.01" class="form-control" id="ovary_volume_left" name="ovary_volume_left" placeholder="e.g., 3-8 cm³ (normal range)">
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <div class="form-check mt-4">
                                                <input class="form-check-input" type="checkbox" id="dominant_follicle_left" name="dominant_follicle_left">
                                                <label class="form-check-label" for="dominant_follicle_left">Dominant Follicle</label>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-md-4 mb-3">
                                    <label for="dominant_follicle_size" class="form-label">Dominant Follicle Size (mm)</label>
                                    <input type="number" step="0.1" class="form-control" id="dominant_follicle_size" name="dominant_follicle_size">
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label for="endometrial_thickness" class="form-label">Endometrial Thickness (mm)</label>
                                    <input type="number" step="0.1" class="form-control" id="endometrial_thickness" name="endometrial_thickness" placeholder="e.g., 7-14 mm (normal range)">
                                </div>
                            </div>
                            
                            <div class="section-header">
                                <h6 class="mb-0">Hormonal Levels</h6>
                            </div>
                            <div class="row">
                                <div class="col-md-3 mb-3">
                                    <label for="fsh_level" class="form-label">FSH Level (mIU/mL)</label>
                                    <input type="number" step="0.01" class="form-control" id="fsh_level" name="fsh_level" placeholder="e.g., 4.7-21.5 mIU/mL">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label for="lh_level" class="form-label">LH Level (mIU/mL)</label>
                                    <input type="number" step="0.01" class="form-control" id="lh_level" name="lh_level" placeholder="e.g., 2.4-12.6 mIU/mL">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label for="amh_level" class="form-label">AMH Level (ng/mL)</label>
                                    <input type="number" step="0.01" class="form-control" id="amh_level" name="amh_level" placeholder="e.g., 1.0-4.0 ng/mL">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label for="estradiol_level" class="form-label">Estradiol (pg/mL)</label>
                                    <input type="number" step="0.01" class="form-control" id="estradiol_level" name="estradiol_level" placeholder="e.g., 30-400 pg/mL">
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-md-3 mb-3">
                                    <label for="progesterone_level" class="form-label">Progesterone (ng/mL)</label>
                                    <input type="number" step="0.01" class="form-control" id="progesterone_level" name="progesterone_level" placeholder="e.g., 0.3-1.5 ng/mL">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label for="testosterone_level" class="form-label">Testosterone (ng/dL)</label>
                                    <input type="number" step="0.01" class="form-control" id="testosterone_level" name="testosterone_level" placeholder="e.g., 15-70 ng/dL">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label for="tsh_level" class="form-label">TSH Level (μIU/mL)</label>
                                    <input type="number" step="0.01" class="form-control" id="tsh_level" name="tsh_level" placeholder="e.g., 0.4-4.0 μIU/mL">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label for="prolactin_level" class="form-label">Prolactin (ng/mL)</label>
                                    <input type="number" step="0.01" class="form-control" id="prolactin_level" name="prolactin_level" placeholder="e.g., 4.8-23.3 ng/mL">
                                </div>
                            </div>
                            
                            <div class="section-header">
                                <h6 class="mb-0">Metabolic Markers</h6>
                            </div>
                            <div class="row">
                                <div class="col-md-4 mb-3">
                                    <label for="fasting_glucose" class="form-label">Fasting Glucose (mg/dL)</label>
                                    <input type="number" step="0.1" class="form-control" id="fasting_glucose" name="fasting_glucose" placeholder="e.g., 70-100 mg/dL">
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label for="fasting_insulin" class="form-label">Fasting Insulin (μIU/mL)</label>
                                    <input type="number" step="0.1" class="form-control" id="fasting_insulin" name="fasting_insulin" placeholder="e.g., 2.6-24.9 μIU/mL">
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label for="hba1c" class="form-label">HbA1c (%)</label>
                                    <input type="number" step="0.1" class="form-control" id="hba1c" name="hba1c" placeholder="e.g., 4.0-5.6%">
                                </div>
                            </div>
                            
                            <div class="section-header">
                                <h6 class="mb-0">PCOS Symptoms</h6>
                            </div>
                            <div class="row">
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Irregular Cycles</label>
                                    <select class="form-select" name="irregular_cycles">
                                        <option value="no">No</option>
                                        <option value="yes">Yes</option>
                                    </select>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Hirsutism</label>
                                    <select class="form-select" name="hirsutism">
                                        <option value="no">No</option>
                                        <option value="yes">Yes</option>
                                    </select>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Acne</label>
                                    <select class="form-select" name="acne">
                                        <option value="no">No</option>
                                        <option value="yes">Yes</option>
                                    </select>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Hair Loss</label>
                                    <select class="form-select" name="hair_loss">
                                        <option value="no">No</option>
                                        <option value="yes">Yes</option>
                                    </select>
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Weight Issues</label>
                                    <select class="form-select" name="weight_issues">
                                        <option value="no">No</option>
                                        <option value="yes">Yes</option>
                                    </select>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Skin Tags</label>
                                    <select class="form-select" name="skin_tags">
                                        <option value="no">No</option>
                                        <option value="yes">Yes</option>
                                    </select>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Dark Skin Patches</label>
                                    <select class="form-select" name="dark_skin_patches">
                                        <option value="no">No</option>
                                        <option value="yes">Yes</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="diagnosis" class="form-label">Diagnosis</label>
                                <textarea class="form-control" id="diagnosis" name="diagnosis" rows="3" placeholder="Enter diagnosis..."></textarea>
                            </div>
                            
                            <div class="mb-3">
                                <label for="treatment_plan" class="form-label">Treatment Plan</label>
                                <textarea class="form-control" id="treatment_plan" name="treatment_plan" rows="3" placeholder="Enter treatment plan..."></textarea>
                            </div>
                            
                            <div class="mb-3">
                                <label for="notes" class="form-label">Additional Notes</label>
                                <textarea class="form-control" id="notes" name="notes" rows="3" placeholder="Any additional observations or notes..."></textarea>
                            </div>
                            
                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-2"></i>Save Record
                                </button>
                                <a href="dashboard.php" class="btn btn-secondary">
                                    <i class="fas fa-times me-2"></i>Cancel
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>