[file name]: doctor/appointments.php
[file content begin]
<?php
require_once '../config.php';
requireRole('doctor');

$db = new Database();
$conn = $db->getConnection();

// Get doctor info
$stmt = $conn->prepare("SELECT * FROM doctors WHERE user_id = ?");
$stmt->execute([$_SESSION['user_id']]);
$doctor = $stmt->fetch();

// Get scheduled/completed/cancelled appointments
$stmt = $conn->prepare("
    SELECT a.*, p.name as patient_name, p.contact as patient_contact 
    FROM appointments a
    JOIN patients p ON a.patient_id = p.id
    WHERE a.doctor_id = ? AND a.status != 'pending'
    ORDER BY a.appointment_date DESC, a.appointment_time DESC
");
$stmt->execute([$doctor['id']]);
$appointments = $stmt->fetchAll();

// Get pending appointment requests
$stmt = $conn->prepare("
    SELECT a.*, p.name as patient_name, p.contact as patient_contact 
    FROM appointments a
    JOIN patients p ON a.patient_id = p.id
    WHERE a.doctor_id = ? AND a.status = 'pending'
    ORDER BY a.appointment_date ASC, a.appointment_time ASC
");
$stmt->execute([$doctor['id']]);
$pending_requests = $stmt->fetchAll();

// Handle status update
if (isset($_GET['update_status'])) {
    $appointment_id = (int)$_GET['id'];
    $status = sanitize($_GET['status']);
    
    try {
        $stmt = $conn->prepare("UPDATE appointments SET status = ? WHERE id = ?");
        $stmt->execute([$status, $appointment_id]);
        
        $success = 'Appointment status updated successfully!';
        logActivity($_SESSION['user_id'], 'Updated appointment status', "Appointment ID: $appointment_id, Status: $status");
        
        // Refresh the page
        header('Location: appointments.php');
        exit();
    } catch (Exception $e) {
        $error = 'Failed to update appointment status: ' . $e->getMessage();
    }
}

// Handle new appointment form
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $patient_id = (int)$_POST['patient_id'];
    $appointment_date = $_POST['appointment_date'];
    $appointment_time = $_POST['appointment_time'];
    $reason = sanitize($_POST['reason']);
    
    try {
        $stmt = $conn->prepare("INSERT INTO appointments 
            (patient_id, doctor_id, appointment_date, appointment_time, reason)
            VALUES (?, ?, ?, ?, ?)");
        
        $stmt->execute([$patient_id, $doctor['id'], $appointment_date, $appointment_time, $reason]);
        
        $success = 'Appointment scheduled successfully!';
        logActivity($_SESSION['user_id'], 'Scheduled appointment', "Patient ID: $patient_id, Date: $appointment_date");
        
        // Refresh the page
        header('Location: appointments.php');
        exit();
    } catch (Exception $e) {
        $error = 'Failed to schedule appointment: ' . $e->getMessage();
    }
}

// Get all patients
$stmt = $conn->query("SELECT * FROM patients ORDER BY name");
$patients = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Appointments - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .status-badge {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        .status-scheduled { background-color: #0d6efd; color: white; }
        .status-completed { background-color: #198754; color: white; }
        .status-cancelled { background-color: #dc3545; color: white; }
        .status-pending { background-color: #ffc107; color: black; }
        .status-no_show { background-color: #6c757d; color: white; }
        
        .appointment-card {
            border-left: 4px solid;
            transition: transform 0.2s;
        }
        .appointment-card:hover {
            transform: translateY(-2px);
        }
        .appointment-card.scheduled { border-left-color: #0d6efd; }
        .appointment-card.completed { border-left-color: #198754; }
        .appointment-card.cancelled { border-left-color: #dc3545; }
        .appointment-card.pending { border-left-color: #ffc107; }
        .appointment-card.no_show { border-left-color: #6c757d; }
    </style>
</head>
<body style="background: linear-gradient(to bottom right, #f0f4f8, #e9f0f7);">
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4" style="background-color: transparent;">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Appointments</h1>
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addAppointmentModal">
                        <i class="fas fa-plus me-2"></i>Schedule Appointment
                    </button>
                </div>
                
                <?php if (isset($success)): ?>
                    <div class="alert alert-success alert-dismissible fade show">
                        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if (isset($error)): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <!-- Pending Appointment Requests -->
                <?php if (!empty($pending_requests)): ?>
                <div class="card mb-4">
                    <div class="card-header bg-warning text-dark">
                        <h5 class="mb-0">
                            <i class="fas fa-clock me-2"></i>
                            Pending Appointment Requests (<?php echo count($pending_requests); ?>)
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Patient</th>
                                        <th>Requested Date</th>
                                        <th>Requested Time</th>
                                        <th>Reason</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($pending_requests as $request): ?>
                                        <tr>
                                            <td>
                                                <div><?php echo safeHtmlspecialchars($request['patient_name']); ?></div>
                                                <small class="text-muted"><?php echo $request['patient_contact']; ?></small>
                                            </td>
                                            <td><?php echo date('M j, Y', strtotime($request['appointment_date'])); ?></td>
                                            <td><?php echo date('g:i A', strtotime($request['appointment_time'])); ?></td>
                                            <td><?php echo nl2br(safeHtmlspecialchars(substr($request['reason'], 0, 50) . (strlen($request['reason'] ?? '') > 50 ? '...' : ''))); ?></td>
                                            <td>
                                                <a href="appointments.php?update_status=1&id=<?php echo $request['id']; ?>&status=scheduled" 
                                                   class="btn btn-sm btn-success" title="Approve">
                                                    <i class="fas fa-check"></i>
                                                </a>
                                                <a href="appointments.php?update_status=1&id=<?php echo $request['id']; ?>&status=cancelled" 
                                                   class="btn btn-sm btn-danger" title="Decline" onclick="return confirm('Are you sure you want to decline this request?');">
                                                    <i class="fas fa-times"></i>
                                                </a>
                                                <button type="button" class="btn btn-sm btn-outline-primary" title="Edit & Approve"
                                                        data-bs-toggle="modal" data-bs-target="#editAppointmentModal<?php echo $request['id']; ?>">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Appointments Cards View -->
                <h3 class="h4">Scheduled Appointments</h3>
                <div class="row mb-4">
                    <?php foreach ($appointments as $appointment): 
                        $status_class = '';
                        if ($appointment['status'] == 'scheduled') $status_class = 'scheduled';
                        elseif ($appointment['status'] == 'completed') $status_class = 'completed';
                        elseif ($appointment['status'] == 'cancelled') $status_class = 'cancelled';
                        elseif ($appointment['status'] == 'pending') $status_class = 'pending';
                        elseif ($appointment['status'] == 'no_show') $status_class = 'no_show';
                    ?>
                        <div class="col-md-6 col-lg-4 mb-3">
                            <div class="card appointment-card <?php echo $status_class; ?>">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between align-items-start mb-2">
                                        <h6 class="card-title mb-0"><?php echo safeHtmlspecialchars($appointment['patient_name']); ?></h6>
                                        <span class="status-badge status-<?php echo $appointment['status']; ?>">
                                            <?php echo ucfirst(str_replace('_', ' ', $appointment['status'])); ?>
                                        </span>
                                    </div>
                                    
                                    <div class="text-muted mb-2">
                                        <i class="fas fa-calendar me-2"></i>
                                        <?php echo date('F j, Y', strtotime($appointment['appointment_date'])); ?>
                                    </div>
                                    
                                    <div class="text-muted mb-2">
                                        <i class="fas fa-clock me-2"></i>
                                        <?php echo date('g:i A', strtotime($appointment['appointment_time'])); ?>
                                    </div>
                                    
                                    <?php if ($appointment['reason']): ?>
                                        <div class="mb-2">
                                            <strong>Reason:</strong>
                                            <p class="mb-0"><?php echo nl2br(safeHtmlspecialchars($appointment['reason'])); ?></p>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <?php if ($appointment['notes']): ?>
                                        <div class="mb-2">
                                            <strong>Notes:</strong>
                                            <p class="mb-0"><?php echo nl2br(safeHtmlspecialchars($appointment['notes'])); ?></p>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div class="d-flex gap-2 mt-3">
                                        <?php if ($appointment['status'] == 'scheduled'): ?>
                                            <a href="appointments.php?update_status=1&id=<?php echo $appointment['id']; ?>&status=completed" 
                                               class="btn btn-sm btn-success">
                                                <i class="fas fa-check"></i> Complete
                                            </a>
                                            <a href="appointments.php?update_status=1&id=<?php echo $appointment['id']; ?>&status=cancelled" 
                                               class="btn btn-sm btn-danger">
                                                <i class="fas fa-times"></i> Cancel
                                            </a>
                                        <?php endif; ?>
                                        
                                        <button type="button" class="btn btn-sm btn-outline-primary" 
                                                data-bs-toggle="modal" data-bs-target="#notesModal<?php echo $appointment['id']; ?>">
                                            <i class="fas fa-edit"></i> Notes
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Notes Modal -->
                        <div class="modal fade" id="notesModal<?php echo $appointment['id']; ?>" tabindex="-1" aria-hidden="true">
                            <div class="modal-dialog">
                                <div class="modal-content">
                                    <div class="modal-header">
                                        <h5 class="modal-title">Add Notes for <?php echo safeHtmlspecialchars($appointment['patient_name']); ?></h5>
                                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                    </div>
                                    <form method="POST" action="update_appointment_notes.php">
                                        <div class="modal-body">
                                            <input type="hidden" name="appointment_id" value="<?php echo $appointment['id']; ?>">
                                            <div class="mb-3">
                                                <label for="notes" class="form-label">Appointment Notes</label>
                                                <textarea class="form-control" id="notes" name="notes" rows="4"><?php echo safeHtmlspecialchars($appointment['notes']); ?></textarea>
                                            </div>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                            <button type="submit" class="btn btn-primary">Save Notes</button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                
                <!-- Appointments Table View (for larger screens) -->
                <div class="card d-none d-lg-block">
                    <div class="card-header">
                        <h6 class="mb-0">All Appointments</h6>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive-lg">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Patient</th>
                                        <th>Date</th>
                                        <th>Time</th>
                                        <th>Reason</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($appointments as $appointment): 
                                        $status_class = '';
                                        if ($appointment['status'] == 'scheduled') $status_class = 'status-scheduled';
                                        elseif ($appointment['status'] == 'completed') $status_class = 'status-completed';
                                        elseif ($appointment['status'] == 'cancelled') $status_class = 'status-cancelled';
                                        elseif ($appointment['status'] == 'pending') $status_class = 'status-pending';
                                        elseif ($appointment['status'] == 'no_show') $status_class = 'status-no_show';
                                    ?>
                                        <tr>
                                            <td>
                                                <div><?php echo safeHtmlspecialchars($appointment['patient_name']); ?></div>
                                                <small class="text-muted"><?php echo $appointment['patient_contact']; ?></small>
                                            </td>
                                            <td><?php echo date('M j, Y', strtotime($appointment['appointment_date'])); ?></td>
                                            <td><?php echo date('g:i A', strtotime($appointment['appointment_time'])); ?></td>
                                            <td><?php echo nl2br(safeHtmlspecialchars(substr($appointment['reason'], 0, 50) . (strlen($appointment['reason'] ?? '') > 50 ? '...' : ''))); ?></td>
                                            <td>
                                                <span class="status-badge <?php echo $status_class; ?>">
                                                    <?php echo ucfirst(str_replace('_', ' ', $appointment['status'])); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <?php if ($appointment['status'] == 'scheduled'): ?>
                                                    <a href="appointments.php?update_status=1&id=<?php echo $appointment['id']; ?>&status=completed" 
                                                       class="btn btn-sm btn-success">
                                                        <i class="fas fa-check"></i>
                                                    </a>
                                                    <a href="appointments.php?update_status=1&id=<?php echo $appointment['id']; ?>&status=cancelled" 
                                                       class="btn btn-sm btn-danger">
                                                        <i class="fas fa-times"></i>
                                                    </a>
                                                <?php endif; ?>
                                                <button type="button" class="btn btn-sm btn-outline-primary" 
                                                        data-bs-toggle="modal" data-bs-target="#notesModal<?php echo $appointment['id']; ?>">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <!-- Add Appointment Modal -->
    <div class="modal fade" id="addAppointmentModal" tabindex="-1" aria-labelledby="addAppointmentModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addAppointmentModalLabel">Schedule New Appointment</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="patient_id" class="form-label">Patient *</label>
                            <select class="form-select" id="patient_id" name="patient_id" required>
                                <option value="">Select Patient</option>
                                <?php foreach ($patients as $patient): ?>
                                    <option value="<?php echo $patient['id']; ?>">
                                        <?php echo safeHtmlspecialchars($patient['name']) . ' (Age: ' . safeHtmlspecialchars($patient['age']) . ')'; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="appointment_date" class="form-label">Date *</label>
                                <input type="date" class="form-control" id="appointment_date" name="appointment_date" required min="<?php echo date('Y-m-d'); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="appointment_time" class="form-label">Time *</label>
                                <input type="time" class="form-control" id="appointment_time" name="appointment_time" required>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="reason" class="form-label">Reason for Appointment *</label>
                            <textarea class="form-control" id="reason" name="reason" rows="3" required placeholder="Enter the reason for this appointment..."></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Schedule Appointment</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
[file content end]