<?php
require_once '../config.php';
requireHasRole('doctor');

$db = new Database();
$conn = $db->getConnection();

// Get the doctor's specific ID from the 'doctors' table
$stmt_doc = $conn->prepare("SELECT id FROM doctors WHERE user_id = ?");
$stmt_doc->execute([$_SESSION['user_id']]);
$doctor = $stmt_doc->fetch();
$doctor_id = $doctor['id'] ?? 0;

// --- Stats for the logged-in doctor ---
$stmt_stats = $conn->prepare("
    SELECT 
        (SELECT COUNT(*) FROM patients WHERE doctor_id = ?) as total_patients,
        (SELECT COUNT(*) FROM appointments WHERE doctor_id = ? AND status = 'scheduled' AND appointment_date >= CURDATE()) as upcoming_appointments,
        (SELECT COUNT(*) FROM appointments WHERE doctor_id = ? AND status = 'pending') as pending_requests,
        (SELECT COUNT(DISTINCT p.id) FROM pcos_comprehensive_assessments pca JOIN patients p ON pca.patient_id = p.id WHERE p.doctor_id = ? AND pca.physician_reviewed = 0) as pending_reviews
    FROM patients p
");
$stmt_stats->execute([$doctor_id, $doctor_id, $doctor_id, $doctor_id]);
$stats = $stmt_stats->fetch();

// --- Fetch upcoming appointments (limit 5) ---
$stmt_appointments = $conn->prepare("
    SELECT a.*, p.name as patient_name
    FROM appointments a
    JOIN patients p ON a.patient_id = p.id
    WHERE a.doctor_id = ? AND a.status = 'scheduled' AND a.appointment_date >= CURDATE()
    ORDER BY a.appointment_date ASC, a.appointment_time ASC
    LIMIT 5
");
$stmt_appointments->execute([$doctor_id]);
$upcoming_appointments = $stmt_appointments->fetchAll();

// --- Fetch recent assessments needing review (limit 5) ---
$stmt_reviews = $conn->prepare("
    SELECT pca.id, pca.created_at, p.name as patient_name, pca.pcos_probability, pca.risk_level
    FROM pcos_comprehensive_assessments pca
    JOIN patients p ON pca.patient_id = p.id
    WHERE p.doctor_id = ? AND pca.physician_reviewed = 0
    ORDER BY pca.created_at DESC
    LIMIT 5
");
$stmt_reviews->execute([$doctor_id]);
$pending_assessments = $stmt_reviews->fetchAll();

// --- Fetch risk distribution for chart ---
$stmt_risk = $conn->prepare("
    SELECT 
        COALESCE(latest_assessment.risk_level, 'Not Assessed') as risk_level, 
        COUNT(p.id) as patient_count
    FROM patients p
    LEFT JOIN (
        SELECT patient_id, risk_level, ROW_NUMBER() OVER(PARTITION BY patient_id ORDER BY created_at DESC) as rn
        FROM pcos_comprehensive_assessments
    ) AS latest_assessment ON p.id = latest_assessment.patient_id AND latest_assessment.rn = 1
    WHERE p.doctor_id = ?
    GROUP BY risk_level
");
$stmt_risk->execute([$doctor_id]);
$risk_data = $stmt_risk->fetchAll();

$risk_chart_labels = array_column($risk_data, 'risk_level');
$risk_chart_data = array_column($risk_data, 'patient_count');


// --- Fetch assigned patients ---
$sql = "
    SELECT p.id, p.name, u.email, u.username, p.subscription_status, p.subscription_end_date, u.created_at
    FROM patients p
    JOIN users u ON p.user_id = u.id
    WHERE p.doctor_id = ?
    ORDER BY p.name ASC
";
$stmt_patients = $conn->prepare($sql);
$stmt_patients->execute([$doctor_id]);
$patients = $stmt_patients->fetchAll();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Doctor Dashboard - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        .stat-card {
            border-left: 5px solid;
            transition: transform 0.2s, box-shadow 0.2s;
        }
        .stat-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        .stat-card .card-title { font-weight: 500; }
        .stat-card .display-4 { font-weight: 700; }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-tachometer-alt me-2"></i>Doctor Dashboard</h1>
                </div>

                <!-- Stats Cards -->
                <div class="row mb-4">
                    <div class="col-xl-3 col-md-6 mb-4">
                        <div class="card stat-card border-primary">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-primary text-uppercase">My Patients</h6>
                                        <div id="stat-total-patients" class="display-4 text-primary"><?php echo $stats['total_patients'] ?? 0; ?></div>
                                    </div>
                                    <i class="fas fa-user-injured fa-3x text-primary opacity-50"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-xl-3 col-md-6 mb-4">
                        <div class="card stat-card border-success">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-success text-uppercase">Upcoming Appointments</h6>
                                        <div id="stat-upcoming-appointments" class="display-4 text-success"><?php echo $stats['upcoming_appointments'] ?? 0; ?></div>
                                    </div>
                                    <i class="fas fa-calendar-check fa-3x text-success opacity-50"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-xl-3 col-md-6 mb-4">
                        <div class="card stat-card border-warning">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-warning text-uppercase">Pending Requests</h6>
                                        <div id="stat-pending-requests" class="display-4 text-warning"><?php echo $stats['pending_requests'] ?? 0; ?></div>
                                    </div>
                                    <i class="fas fa-clock fa-3x text-warning opacity-50"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-xl-3 col-md-6 mb-4">
                        <div class="card stat-card border-danger">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-danger text-uppercase">Assessments to Review</h6>
                                        <div id="stat-pending-reviews" class="display-4 text-danger"><?php echo $stats['pending_reviews'] ?? 0; ?></div>
                                    </div>
                                    <i class="fas fa-clipboard-check fa-3x text-danger opacity-50"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <!-- Upcoming Appointments & Risk Distribution Chart -->
                    <div class="col-lg-7 mb-4">
                        <div class="card h-100">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-calendar-alt me-2 text-success"></i>Upcoming Appointments</h5>
                            </div>
                            <div id="upcoming-appointments-list" class="card-body">
                                <?php if (empty($upcoming_appointments)): ?>
                                    <p class="text-muted text-center">No upcoming appointments.</p>
                                <?php else: ?>
                                    <ul class="list-group list-group-flush">
                                        <?php foreach ($upcoming_appointments as $app): ?>
                                            <li class="list-group-item d-flex justify-content-between align-items-center">
                                                <div>
                                                    <strong><?php echo htmlspecialchars($app['patient_name']); ?></strong><br>
                                                    <small class="text-muted"><?php echo date('D, M j, Y', strtotime($app['appointment_date'])); ?> at <?php echo date('g:i A', strtotime($app['appointment_time'])); ?></small>
                                                </div>
                                                <a href="appointments.php" class="btn btn-sm btn-outline-primary">View</a>
                                            </li>
                                        <?php endforeach; ?>
                                    </ul>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-5 mb-4">
                        <div class="card h-100">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-chart-pie me-2 text-primary"></i>Patient Risk Distribution</h5>
                            </div>
                            <div class="card-body d-flex align-items-center justify-content-center">
                                <canvas id="riskDistributionChart"></canvas>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-12 mb-4">
                        <div class="card h-100">
                            <div class="card-header bg-danger text-white">
                                <h5 class="mb-0"><i class="fas fa-exclamation-triangle me-2"></i>Action Required: Assessments to Review</h5>
                            </div>
                            <div id="pending-assessments-list" class="card-body">
                                <?php if (empty($pending_assessments)): ?>
                                    <p class="text-muted text-center">No new assessments require your review.</p>
                                <?php else: ?>
                                    <ul class="list-group list-group-flush">
                                        <?php foreach ($pending_assessments as $assess): ?>
                                            <li class="list-group-item d-flex justify-content-between align-items-center">
                                                <div>
                                                    <strong><?php echo htmlspecialchars($assess['patient_name']); ?></strong><br>
                                                    <small class="text-muted">Submitted: <?php echo date('M j, Y', strtotime($assess['created_at'])); ?> | Risk: <span class="fw-bold text-danger"><?php echo htmlspecialchars($assess['risk_level']); ?></span></small>
                                                </div>
                                                <a href="view_comprehensive_assessment.php?id=<?php echo $assess['id']; ?>" class="btn btn-sm btn-outline-danger">Review</a>
                                            </li>
                                        <?php endforeach; ?>
                                    </ul>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Patient List -->
                <div class="card mt-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0"><i class="fas fa-users me-2"></i>My Patient List</h5>
                        <a href="patients.php" class="btn btn-sm btn-outline-secondary">View All</a>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th>Name</th>
                                        <th>Email</th>
                                        <th>Subscription</th>
                                        <th class="text-center">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($patients)): ?>
                                        <tr><td colspan="4" class="text-center text-muted p-4">You have no patients assigned to you.</td></tr>
                                    <?php else: ?>
                                        <?php foreach (array_slice($patients, 0, 10) as $patient): // Show first 10 ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($patient['name']); ?></td>
                                                <td><?php echo htmlspecialchars($patient['email']); ?></td>
                                                <td><span class="badge bg-<?php echo $patient['subscription_status'] === 'active' ? 'success' : 'secondary'; ?>"><?php echo ucfirst($patient['subscription_status'] ?? 'inactive'); ?></span></td>
                                                <td class="text-center">
                                                    <a href="patient_profile.php?id=<?php echo $patient['id']; ?>" class="btn btn-sm btn-outline-primary" title="View Profile"><i class="fas fa-user"></i></a>
                                                    <a href="add_record.php?patient_id=<?php echo $patient['id']; ?>" class="btn btn-sm btn-outline-info" title="New Record"><i class="fas fa-file-plus"></i></a>
                                                    <a href="schedule_appointment.php?patient_id=<?php echo $patient['id']; ?>" class="btn btn-sm btn-outline-success" title="New Appointment"><i class="fas fa-calendar-plus"></i></a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    let riskDistributionChart;

    document.addEventListener('DOMContentLoaded', function() {
        const riskChartCtx = document.getElementById('riskDistributionChart');
        if (riskChartCtx) {
            const riskLabels = <?php echo json_encode($risk_chart_labels); ?>;
            const riskData = <?php echo json_encode($risk_chart_data); ?>;

            // Define colors for each risk level
            const riskColors = {
                'High': 'rgba(220, 53, 69, 0.7)',
                'Moderate-High': 'rgba(253, 126, 20, 0.7)',
                'Moderate': 'rgba(255, 193, 7, 0.7)',
                'Low': 'rgba(25, 135, 84, 0.7)',
                'Not Assessed': 'rgba(108, 117, 125, 0.7)'
            };

            const backgroundColors = riskLabels.map(label => riskColors[label] || riskColors['Not Assessed']);

            riskDistributionChart = new Chart(riskChartCtx, {
                type: 'doughnut',
                data: {
                    labels: riskLabels,
                    datasets: [{
                        label: 'Patient Count',
                        data: riskData,
                        backgroundColor: backgroundColors,
                        borderColor: '#fff',
                        borderWidth: 2
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                        }
                    }
                }
            });
        }

        // --- Real-time update logic ---
        function updateDashboard() {
            fetch('get_dashboard_data.php')
                .then(response => response.json())
                .then(data => {
                    if (data.error) {
                        console.error('Error fetching dashboard data:', data.error);
                        return;
                    }

                    // Update stats cards
                    document.getElementById('stat-total-patients').textContent = data.stats.total_patients || 0;
                    document.getElementById('stat-upcoming-appointments').textContent = data.stats.upcoming_appointments || 0;
                    document.getElementById('stat-pending-requests').textContent = data.stats.pending_requests || 0;
                    document.getElementById('stat-pending-reviews').textContent = data.stats.pending_reviews || 0;

                    // Update lists
                    document.getElementById('upcoming-appointments-list').innerHTML = data.upcoming_appointments_html;
                    document.getElementById('pending-assessments-list').innerHTML = data.pending_assessments_html;

                    // Update chart
                    if (riskDistributionChart && data.risk_chart_data) {
                        riskDistributionChart.data.labels = data.risk_chart_data.labels;
                        riskDistributionChart.data.datasets[0].data = data.risk_chart_data.data;
                        
                        const riskColors = { 'High': 'rgba(220, 53, 69, 0.7)', 'Moderate-High': 'rgba(253, 126, 20, 0.7)', 'Moderate': 'rgba(255, 193, 7, 0.7)', 'Low': 'rgba(25, 135, 84, 0.7)', 'Not Assessed': 'rgba(108, 117, 125, 0.7)' };
                        riskDistributionChart.data.datasets[0].backgroundColor = data.risk_chart_data.labels.map(label => riskColors[label] || riskColors['Not Assessed']);
                        
                        riskDistributionChart.update();
                    }
                })
                .catch(error => console.error('Failed to update dashboard:', error));
        }

        // Poll for updates every 30 seconds
        setInterval(updateDashboard, 30000);
    });
    </script>
</body>
</html>