<?php
require_once '../config.php';
requireHasRole('doctor');

$db = new Database();
$conn = $db->getConnection();

$response = [
    'stats' => [],
    'upcoming_appointments_html' => '',
    'pending_assessments_html' => '',
    'risk_chart_data' => ['labels' => [], 'data' => []]
];

try {
    // Get the doctor's specific ID from the 'doctors' table
    $stmt_doc = $conn->prepare("SELECT id FROM doctors WHERE user_id = ?");
    $stmt_doc->execute([$_SESSION['user_id']]);
    $doctor = $stmt_doc->fetch();
    $doctor_id = $doctor['id'] ?? 0;

    // --- Stats for the logged-in doctor ---
    $stmt_stats = $conn->prepare("
        SELECT 
            (SELECT COUNT(*) FROM patients WHERE doctor_id = :doc_id1) as total_patients,
            (SELECT COUNT(*) FROM appointments WHERE doctor_id = :doc_id2 AND status = 'scheduled' AND appointment_date >= CURDATE()) as upcoming_appointments,
            (SELECT COUNT(*) FROM appointments WHERE doctor_id = :doc_id3 AND status = 'pending') as pending_requests,
            (SELECT COUNT(DISTINCT p.id) FROM pcos_comprehensive_assessments pca JOIN patients p ON pca.patient_id = p.id WHERE p.doctor_id = :doc_id4 AND pca.physician_reviewed = 0) as pending_reviews
    ");
    $stmt_stats->execute(['doc_id1' => $doctor_id, 'doc_id2' => $doctor_id, 'doc_id3' => $doctor_id, 'doc_id4' => $doctor_id]);
    $response['stats'] = $stmt_stats->fetch(PDO::FETCH_ASSOC);

    // --- Fetch upcoming appointments (limit 5) ---
    $stmt_appointments = $conn->prepare("
        SELECT a.*, p.name as patient_name
        FROM appointments a
        JOIN patients p ON a.patient_id = p.id
        WHERE a.doctor_id = ? AND a.status = 'scheduled' AND a.appointment_date >= CURDATE()
        ORDER BY a.appointment_date ASC, a.appointment_time ASC
        LIMIT 5
    ");
    $stmt_appointments->execute([$doctor_id]);
    $upcoming_appointments = $stmt_appointments->fetchAll();

    if (empty($upcoming_appointments)) {
        $response['upcoming_appointments_html'] = '<p class="text-muted text-center">No upcoming appointments.</p>';
    } else {
        $html = '<ul class="list-group list-group-flush">';
        foreach ($upcoming_appointments as $app) {
            $html .= '<li class="list-group-item d-flex justify-content-between align-items-center">
                        <div>
                            <strong>' . htmlspecialchars($app['patient_name']) . '</strong><br>
                            <small class="text-muted">' . date('D, M j, Y', strtotime($app['appointment_date'])) . ' at ' . date('g:i A', strtotime($app['appointment_time'])) . '</small>
                        </div>
                        <a href="appointments.php" class="btn btn-sm btn-outline-primary">View</a>
                      </li>';
        }
        $html .= '</ul>';
        $response['upcoming_appointments_html'] = $html;
    }

    // --- Fetch recent assessments needing review (limit 5) ---
    $stmt_reviews = $conn->prepare("
        SELECT pca.id, pca.created_at, p.name as patient_name, pca.pcos_probability, pca.risk_level
        FROM pcos_comprehensive_assessments pca
        JOIN patients p ON pca.patient_id = p.id
        WHERE p.doctor_id = ? AND pca.physician_reviewed = 0
        ORDER BY pca.created_at DESC
        LIMIT 5
    ");
    $stmt_reviews->execute([$doctor_id]);
    $pending_assessments = $stmt_reviews->fetchAll();

    if (empty($pending_assessments)) {
        $response['pending_assessments_html'] = '<p class="text-muted text-center">No new assessments require your review.</p>';
    } else {
        $html = '<ul class="list-group list-group-flush">';
        foreach ($pending_assessments as $assess) {
            $html .= '<li class="list-group-item d-flex justify-content-between align-items-center">
                        <div>
                            <strong>' . htmlspecialchars($assess['patient_name']) . '</strong><br>
                            <small class="text-muted">Submitted: ' . date('M j, Y', strtotime($assess['created_at'])) . ' | Risk: <span class="fw-bold text-danger">' . htmlspecialchars($assess['risk_level']) . '</span></small>
                        </div>
                        <a href="view_comprehensive_assessment.php?id=' . $assess['id'] . '" class="btn btn-sm btn-outline-danger">Review</a>
                      </li>';
        }
        $html .= '</ul>';
        $response['pending_assessments_html'] = $html;
    }

    // --- Fetch risk distribution for chart ---
    $stmt_risk = $conn->prepare("
        SELECT COALESCE(latest_assessment.risk_level, 'Not Assessed') as risk_level, COUNT(p.id) as patient_count
        FROM patients p
        LEFT JOIN (
            SELECT patient_id, risk_level, ROW_NUMBER() OVER(PARTITION BY patient_id ORDER BY created_at DESC) as rn
            FROM pcos_comprehensive_assessments
        ) AS latest_assessment ON p.id = latest_assessment.patient_id AND latest_assessment.rn = 1
        WHERE p.doctor_id = ?
        GROUP BY risk_level
    ");
    $stmt_risk->execute([$doctor_id]);
    $risk_data = $stmt_risk->fetchAll();
    $response['risk_chart_data']['labels'] = array_column($risk_data, 'risk_level');
    $response['risk_chart_data']['data'] = array_column($risk_data, 'patient_count');

} catch (Exception $e) {
    http_response_code(500);
    $response['error'] = $e->getMessage();
}

header('Content-Type: application/json');
echo json_encode($response);