<?php
require_once '../config.php';
requireRole('patient');

$db = new Database();
$conn = $db->getConnection();

// Check if the phone column exists in the patients table
$stmt = $conn->prepare("SHOW COLUMNS FROM patients LIKE 'phone'");
$stmt->execute();
$phone_column_exists = $stmt->fetch() !== false;

// Get patient info - dynamically build query based on available columns
$columns = "p.id, p.user_id, p.name, p.date_of_birth, p.address, p.height, p.weight, p.updated_at, u.email, u.created_at as account_created";
if ($phone_column_exists) {
    $columns .= ", p.phone";
}

$stmt = $conn->prepare("
    SELECT $columns
    FROM patients p 
    JOIN users u ON p.user_id = u.id 
    WHERE u.id = ?
");
$stmt->execute([$_SESSION['user_id']]);
$patient = $stmt->fetch();

// Initialize default values for potentially missing fields
$default_values = [
    'date_of_birth' => '',
    'address' => '',
    'height' => null,
    'weight' => null,
    'name' => '',
    'email' => '',
    'account_created' => date('Y-m-d H:i:s')
];

if ($phone_column_exists) {
    $default_values['phone'] = '';
}

$patient = array_merge($default_values, $patient ?: []);

// Handle form submission
$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Validate input
        $name = trim($_POST['name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $phone = $phone_column_exists ? trim($_POST['phone'] ?? '') : '';
        $address = trim($_POST['address'] ?? '');
        $date_of_birth = $_POST['date_of_birth'] ?? '';
        $height = !empty($_POST['height']) ? (float)$_POST['height'] : null;
        $weight = !empty($_POST['weight']) ? (float)$_POST['weight'] : null;
        
        if (empty($name)) {
            throw new Exception('Name is required');
        }
        
        if (empty($email)) {
            throw new Exception('Email is required');
        }
        
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new Exception('Invalid email format');
        }
        
        // Check if email already exists (excluding current user)
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
        $stmt->execute([$email, $_SESSION['user_id']]);
        if ($stmt->fetch()) {
            throw new Exception('Email already exists');
        }
        
        // Update user table
        $stmt = $conn->prepare("UPDATE users SET email = ? WHERE id = ?");
        $stmt->execute([$email, $_SESSION['user_id']]);
        
        // Build dynamic UPDATE query for patients table
        $update_fields = "name = ?, address = ?, date_of_birth = ?, height = ?, weight = ?, updated_at = NOW()";
        $params = [$name, $address, $date_of_birth, $height, $weight];
        
        if ($phone_column_exists) {
            $update_fields .= ", phone = ?";
            $params[] = $phone;
        }
        
        $params[] = $_SESSION['user_id']; // WHERE condition
        
        $stmt = $conn->prepare("
            UPDATE patients 
            SET $update_fields
            WHERE user_id = ?
        ");
        $stmt->execute($params);
        
        // Handle password change if provided
        if (!empty($_POST['password'])) {
            if ($_POST['password'] !== $_POST['confirm_password']) {
                throw new Exception('Passwords do not match');
            }
            
            if (strlen($_POST['password']) < 6) {
                throw new Exception('Password must be at least 6 characters long');
            }
            
            $hashed_password = password_hash($_POST['password'], PASSWORD_DEFAULT);
            $stmt = $conn->prepare("UPDATE users SET password = ? WHERE id = ?");
            $stmt->execute([$hashed_password, $_SESSION['user_id']]);
        }
        
        $success_message = 'Profile updated successfully!';
        
        // Refresh patient data
        $stmt = $conn->prepare("
            SELECT $columns
            FROM patients p 
            JOIN users u ON p.user_id = u.id 
            WHERE u.id = ?
        ");
        $stmt->execute([$_SESSION['user_id']]);
        $patient = $stmt->fetch();
        
        // Re-initialize default values
        $patient = array_merge($default_values, $patient ?: []);
        
    } catch (Exception $e) {
        $error_message = $e->getMessage();
    }
}

// Calculate BMI if height and weight are available
$bmi = null;
if ($patient['height'] && $patient['weight']) {
    $height_in_meters = $patient['height'] / 100;
    $bmi = round($patient['weight'] / ($height_in_meters * $height_in_meters), 1);
}

// Get BMI category
$bmi_category = '';
if ($bmi !== null) {
    if ($bmi < 18.5) {
        $bmi_category = 'Underweight';
    } elseif ($bmi < 25) {
        $bmi_category = 'Normal weight';
    } elseif ($bmi < 30) {
        $bmi_category = 'Overweight';
    } else {
        $bmi_category = 'Obesity';
    }
}

// Calculate age
$age = 'N/A';
if (!empty($patient['date_of_birth'])) {
    $age = floor((time() - strtotime($patient['date_of_birth'])) / 31556926);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($patient['name']); ?> - Patient Profile - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .profile-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem;
            border-radius: 10px;
            margin-bottom: 2rem;
        }
        .stat-card {
            border-left: 4px solid;
            transition: transform 0.2s;
        }
        .stat-card:hover {
            transform: translateY(-2px);
        }
        .stat-card.primary { border-left-color: #0d6efd; }
        .stat-card.success { border-left-color: #198754; }
        .stat-card.warning { border-left-color: #ffc107; }
        .stat-card.danger { border-left-color: #dc3545; }
        
        .risk-badge {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        .risk-high { background-color: #dc3545; color: white; }
        .risk-moderate { background-color: #fd7e14; color: white; }
        .risk-low { background-color: #ffc107; color: black; }
        .risk-minimal { background-color: #198754; color: white; }
        
        .symptom-badge {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        .symptom-present { background-color: #dc3545; color: white; }
        .symptom-absent { background-color: #198754; color: white; }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <!-- Patient Header -->
                <div class="profile-header">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h2><?php echo htmlspecialchars($patient['name']); ?></h2>
                            <p class="mb-1">Patient ID: <?php echo str_pad($patient['id'], 6, '0', STR_PAD_LEFT); ?></p>
                            <p class="mb-1">Age: <?php echo $patient['age']; ?> years | Gender: <?php echo ucfirst($patient['gender']); ?></p>
                            <p class="mb-0">Contact: <?php echo $patient['contact']; ?></p>
                        </div>
                        <div class="col-md-4 text-end">
                            <div class="btn-group">
                                <a href="add_record.php?patient_id=<?php echo $patient['id']; ?>" class="btn btn-light">
                                    <i class="fas fa-file-medical me-2"></i>New Record
                                </a>
                                <a href="edit_patient.php?id=<?php echo $patient['id']; ?>" class="btn btn-light">
                                    <i class="fas fa-edit me-2"></i>Edit Profile
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Statistics Cards -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card stat-card primary">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted mb-0">Total Records</h6>
                                        <h4 class="fw-bold mb-0"><?php echo count($records); ?></h4>
                                    </div>
                                    <div class="icon-shape bg-primary text-white rounded-circle p-3">
                                        <i class="fas fa-file-medical"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="card stat-card <?php echo ($latest_record && $latest_record['pcos_risk_level'] == 'High') ? 'danger' : 
                                                     (($latest_record && $latest_record['pcos_risk_level'] == 'Moderate') ? 'warning' : 'success'); ?>">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted mb-0">PCOS Risk</h6>
                                        <h4 class="fw-bold mb-0">
                                            <?php echo $latest_record ? $latest_record['pcos_risk_level'] : 'N/A'; ?>
                                        </h4>
                                    </div>
                                    <div class="icon-shape 
                                        <?php echo ($latest_record && $latest_record['pcos_risk_level'] == 'High') ? 'bg-danger' : 
                                                (($latest_record && $latest_record['pcos_risk_level'] == 'Moderate') ? 'bg-warning' : 'bg-success'); ?> 
                                        text-white rounded-circle p-3">
                                        <i class="fas fa-heartbeat"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="card stat-card info">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted mb-0">Active Treatments</h6>
                                        <h4 class="fw-bold mb-0">
                                            <?php 
                                                $active_treatments = array_filter($treatments, function($t) { 
                                                    return $t['status'] == 'active'; 
                                                });
                                                echo count($active_treatments);
                                            ?>
                                        </h4>
                                    </div>
                                    <div class="icon-shape bg-info text-white rounded-circle p-3">
                                        <i class="fas fa-pills"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="card stat-card secondary">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted mb-0">Upcoming Appointments</h6>
                                        <h4 class="fw-bold mb-0">
                                            <?php 
                                                $upcoming = array_filter($appointments, function($a) { 
                                                    return $a['status'] == 'scheduled' && 
                                                           strtotime($a['appointment_date']) >= strtotime('today'); 
                                                });
                                                echo count($upcoming);
                                            ?>
                                        </h4>
                                    </div>
                                    <div class="icon-shape bg-secondary text-white rounded-circle p-3">
                                        <i class="fas fa-calendar-check"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <!-- Patient Details -->
                    <div class="col-md-4 mb-4">
                        <div class="card">
                            <div class="card-header">
                                <h6 class="mb-0">Patient Information</h6>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <strong>Age:</strong> <?php echo $patient['age']; ?> years
                                </div>
                                <div class="mb-3">
                                    <strong>Gender:</strong> <?php echo ucfirst($patient['gender']); ?>
                                </div>
                                <div class="mb-3">
                                    <strong>Contact:</strong> <?php echo $patient['contact']; ?>
                                </div>
                                <?php if ($patient['height'] && $patient['weight']): ?>
                                    <div class="mb-3">
                                        <strong>Height:</strong> <?php echo $patient['height']; ?> cm
                                    </div>
                                    <div class="mb-3">
                                        <strong>Weight:</strong> <?php echo $patient['weight']; ?> kg
                                    </div>
                                    <div class="mb-3">
                                        <strong>BMI:</strong> <?php echo number_format($patient['bmi'], 1); ?>
                                    </div>
                                <?php endif; ?>
                                <?php if ($patient['blood_type']): ?>
                                    <div class="mb-3">
                                        <strong>Blood Type:</strong> <?php echo $patient['blood_type']; ?>
                                    </div>
                                <?php endif; ?>
                                <?php if ($patient['menstrual_cycle_regular']): ?>
                                    <div class="mb-3">
                                        <strong>Menstrual Cycle:</strong> <?php echo ucfirst($patient['menstrual_cycle_regular']); ?>
                                    </div>
                                <?php endif; ?>
                                <?php if ($patient['last_menstrual_period']): ?>
                                    <div class="mb-3">
                                        <strong>Last Period:</strong> <?php echo date('M j, Y', strtotime($patient['last_menstrual_period'])); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Medical History -->
                        <?php if ($patient['medical_history']): ?>
                            <div class="card mt-4">
                                <div class="card-header">
                                    <h6 class="mb-0">Medical History</h6>
                                </div>
                                <div class="card-body">
                                    <p><?php echo nl2br(htmlspecialchars($patient['medical_history'])); ?></p>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- PCOS Symptoms -->
                    <div class="col-md-4 mb-4">
                        <div class="card">
                            <div class="card-header">
                                <h6 class="mb-0">PCOS Symptoms</h6>
                            </div>
                            <div class="card-body">
                                <?php if (!empty($symptoms)): ?>
                                    <div class="row">
                                        <div class="col-6 mb-2">
                                            <span>Irregular Cycles:</span>
                                            <span class="symptom-badge <?php echo $symptoms['irregular_cycles'] == 'yes' ? 'symptom-present' : 'symptom-absent'; ?> float-end">
                                                <?php echo ucfirst($symptoms['irregular_cycles']); ?>
                                            </span>
                                        </div>
                                        <div class="col-6 mb-2">
                                            <span>Hirsutism:</span>
                                            <span class="symptom-badge <?php echo $symptoms['hirsutism'] == 'yes' ? 'symptom-present' : 'symptom-absent'; ?> float-end">
                                                <?php echo ucfirst($symptoms['hirsutism']); ?>
                                            </span>
                                        </div>
                                        <div class="col-6 mb-2">
                                            <span>Acne:</span>
                                            <span class="symptom-badge <?php echo $symptoms['acne'] == 'yes' ? 'symptom-present' : 'symptom-absent'; ?> float-end">
                                                <?php echo ucfirst($symptoms['acne']); ?>
                                            </span>
                                        </div>
                                        <div class="col-6 mb-2">
                                            <span>Hair Loss:</span>
                                            <span class="symptom-badge <?php echo $symptoms['hair_loss'] == 'yes' ? 'symptom-present' : 'symptom-absent'; ?> float-end">
                                                <?php echo ucfirst($symptoms['hair_loss']); ?>
                                            </span>
                                        </div>
                                        <div class="col-6 mb-2">
                                            <span>Weight Issues:</span>
                                            <span class="symptom-badge <?php echo $symptoms['weight_issues'] == 'yes' ? 'symptom-present' : 'symptom-absent'; ?> float-end">
                                                <?php echo ucfirst($symptoms['weight_issues']); ?>
                                            </span>
                                        </div>
                                        <div class="col-6 mb-2">
                                            <span>Skin Tags:</span>
                                            <span class="symptom-badge <?php echo $symptoms['skin_tags'] == 'yes' ? 'symptom-present' : 'symptom-absent'; ?> float-end">
                                                <?php echo ucfirst($symptoms['skin_tags']); ?>
                                            </span>
                                        </div>
                                        <div class="col-6 mb-2">
                                            <span>Dark Skin Patches:</span>
                                            <span class="symptom-badge <?php echo $symptoms['dark_skin_patches'] == 'yes' ? 'symptom-present' : 'symptom-absent'; ?> float-end">
                                                <?php echo ucfirst($symptoms['dark_skin_patches']); ?>
                                            </span>
                                        </div>
                                    </div>
                                <?php else: ?>
                                    <p class="text-muted text-center mb-0">No symptom data recorded.</p>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Latest Record Summary -->
                        <?php if ($latest_record): ?>
                            <div class="card mt-4">
                                <div class="card-header">
                                    <h6 class="mb-0">Latest Examination (<?php echo date('M j, Y', strtotime($latest_record['examination_date'])); ?>)</h6>
                                </div>
                                <div class="card-body">
                                    <div class="row">
                                        <div class="col-6">
                                            <strong>Right Ovary:</strong><br>
                                            <?php echo $latest_record['follicle_count_right'] ? $latest_record['follicle_count_right'] . ' follicles' : 'N/A'; ?>
                                        </div>
                                        <div class="col-6">
                                            <strong>Left Ovary:</strong><br>
                                            <?php echo $latest_record['follicle_count_left'] ? $latest_record['follicle_count_left'] . ' follicles' : 'N/A'; ?>
                                        </div>
                                    </div>
                                    
                                    <?php if ($latest_record['amh_level']): ?>
                                        <div class="mt-2">
                                            <strong>AMH:</strong> <?php echo number_format($latest_record['amh_level'], 2); ?> ng/mL
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div class="mt-2">
                                        <strong>PCOS Score:</strong> <?php echo $latest_record['pcos_score']; ?>
                                        <span class="risk-badge risk-<?php echo strtolower($latest_record['pcos_risk_level']); ?> float-end">
                                            <?php echo $latest_record['pcos_risk_level']; ?> Risk
                                        </span>
                                    </div>
                                    
                                    <div class="mt-3 text-center">
                                        <a href="view_record.php?id=<?php echo $latest_record['id']; ?>" class="btn btn-sm btn-outline-primary">
                                            <i class="fas fa-eye me-1"></i> View Full Details
                                        </a>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Recent Activity -->
                    <div class="col-md-4 mb-4">
                        <!-- Active Treatments -->
                        <div class="card">
                            <div class="card-header">
                                <h6 class="mb-0">Active Treatments</h6>
                            </div>
                            <div class="card-body">
                                <?php if ($active_treatments): ?>
                                    <?php foreach ($active_treatments as $treatment): ?>
                                        <div class="mb-3 pb-3 border-bottom">
                                            <div class="d-flex justify-content-between align-items-start">
                                                <h6 class="mb-1"><?php echo ucfirst($treatment['treatment_type']); ?></h6>
                                                <span class="badge bg-success">Active</span>
                                            </div>
                                            <?php if ($treatment['medication_name']): ?>
                                                <p class="mb-1"><?php echo htmlspecialchars($treatment['medication_name']); ?></p>
                                            <?php endif; ?>
                                            <small class="text-muted">
                                                Started: <?php echo date('M j, Y', strtotime($treatment['start_date'])); ?>
                                                <?php if ($treatment['end_date']): ?>
                                                    | Until: <?php echo date('M j, Y', strtotime($treatment['end_date'])); ?>
                                                <?php endif; ?>
                                            </small>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <p class="text-muted text-center mb-0">No active treatments.</p>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Upcoming Appointments -->
                        <div class="card mt-4">
                            <div class="card-header">
                                <h6 class="mb-0">Upcoming Appointments</h6>
                            </div>
                            <div class="card-body">
                                <?php if ($upcoming): ?>
                                    <?php foreach ($upcoming as $appointment): ?>
                                        <div class="mb-3 pb-3 border-bottom">
                                            <div class="d-flex justify-content-between align-items-start">
                                                <h6 class="mb-1"><?php echo date('M j, Y', strtotime($appointment['appointment_date'])); ?></h6>
                                                <span class="badge bg-primary">Scheduled</span>
                                            </div>
                                            <p class="mb-1"><?php echo date('g:i A', strtotime($appointment['appointment_time'])); ?></p>
                                            <?php if ($appointment['reason']): ?>
                                                <p class="mb-1"><?php echo nl2br(htmlspecialchars($appointment['reason'])); ?></p>
                                            <?php endif; ?>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <p class="text-muted text-center mb-0">No upcoming appointments.</p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Full Records Table -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h6 class="mb-0">Examination History</h6>
                    </div>
                    <div class="card-body">
                        <?php if ($records): ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Date</th>
                                            <th>Doctor</th>
                                            <th>Follicles (R/L)</th>
                                            <th>AMH Level</th>
                                            <th>PCOS Score</th>
                                            <th>Risk Level</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($records as $record): 
                                            $risk_class = '';
                                            if ($record['pcos_risk_level'] == 'High') $risk_class = 'risk-high';
                                            elseif ($record['pcos_risk_level'] == 'Moderate') $risk_class = 'risk-moderate';
                                            elseif ($record['pcos_risk_level'] == 'Low') $risk_class = 'risk-low';
                                            elseif ($record['pcos_risk_level'] == 'Minimal') $risk_class = 'risk-minimal';
                                        ?>
                                            <tr>
                                                <td><?php echo date('M j, Y', strtotime($record['examination_date'])); ?></td>
                                                <td><?php echo htmlspecialchars($record['doctor_name']); ?></td>
                                                <td>
                                                    <?php echo $record['follicle_count_right'] ? $record['follicle_count_right'] : '0'; ?> /
                                                    <?php echo $record['follicle_count_left'] ? $record['follicle_count_left'] : '0'; ?>
                                                </td>
                                                <td><?php echo $record['amh_level'] ? number_format($record['amh_level'], 2) . ' ng/mL' : 'N/A'; ?></td>
                                                <td><?php echo $record['pcos_score'] ? $record['pcos_score'] : 'N/A'; ?></td>
                                                <td>
                                                    <?php if ($record['pcos_risk_level']): ?>
                                                        <span class="risk-badge <?php echo $risk_class; ?>">
                                                            <?php echo $record['pcos_risk_level']; ?>
                                                        </span>
                                                    <?php else: ?>
                                                        N/A
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <a href="view_record.php?id=<?php echo $record['id']; ?>" class="btn btn-sm btn-outline-primary">
                                                        <i class="fas fa-eye"></i> View
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <p class="text-muted text-center mb-0">No examination records found.</p>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
[file content end]