<?php
require_once '../config.php';
requireRole('doctor');

$db = new Database();
$conn = $db->getConnection();

$patient_id = isset($_GET['patient_id']) ? (int)$_GET['patient_id'] : 0;

if (!$patient_id) {
    die('Invalid patient ID.');
}

// Fetch patient details
$stmt = $conn->prepare("SELECT * FROM patients WHERE id = ?");
$stmt->execute([$patient_id]);
$patient = $stmt->fetch();

if (!$patient) {
    die('Patient not found.');
}

$success = '';
$error = '';

// Handle resource assignment
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['assign_resources'])) {
    $resource_ids = $_POST['resource_ids'] ?? [];
    
    if (empty($resource_ids)) {
        $error = "Please select at least one resource to assign.";
    } else {
        $assigned_count = 0;
        $assigned_resource_titles = [];
        foreach ($resource_ids as $resource_id) {
            $resource_id = (int)$resource_id;
            
            // Check if already assigned
            $stmt = $conn->prepare("SELECT id FROM patient_resource_access WHERE patient_id = ? AND resource_id = ?");
            $stmt->execute([$patient_id, $resource_id]);
            
            if (!$stmt->fetch()) {
                $stmt = $conn->prepare("INSERT INTO patient_resource_access (patient_id, resource_id, completion_status) VALUES (?, ?, 'assigned')");
                $stmt->execute([$patient_id, $resource_id]);
                $assigned_count++;

                // Get resource title for notification
                $stmt_title = $conn->prepare("SELECT title FROM pcos_education_resources WHERE id = ?");
                $stmt_title->execute([$resource_id]);
                if ($resource_title = $stmt_title->fetchColumn()) {
                    $assigned_resource_titles[] = $resource_title;
                }
            }
        }
        
        if ($assigned_count > 0) {
            $success = "$assigned_count new resource(s) assigned successfully.";

            // Send notification(s) to the patient
            $doctor_user_id = $_SESSION['user_id'];
            $patient_user_id = $patient['user_id'];
            foreach ($assigned_resource_titles as $title) {
                $message = "Your doctor assigned a new resource: \"$title\"";
                createNotification($patient_user_id, 'message', $message, 'patient/resources.php', $doctor_user_id);
            }
        } else {
            $error = "No new resources were assigned. They may already have access.";
        }
    }
}

// Fetch patient's resource access history
$stmt = $conn->prepare("
    SELECT pra.*, per.title, per.category, per.difficulty_level
    FROM patient_resource_access pra
    JOIN pcos_education_resources per ON pra.resource_id = per.id
    WHERE pra.patient_id = ?
    ORDER BY pra.access_date DESC
");
$stmt->execute([$patient_id]);
$accessed_resources = $stmt->fetchAll();

// Fetch all available resources for assignment
$stmt = $conn->query("SELECT id, title, category FROM pcos_education_resources WHERE is_active = 1 ORDER BY category, title");
$all_resources = $stmt->fetchAll();

// Calculate stats
$total_accessed = count($accessed_resources);
$total_completed = 0;
foreach ($accessed_resources as $res) {
    if ($res['completion_status'] == 'completed') {
        $total_completed++;
    }
}
$completion_rate = $total_accessed > 0 ? round(($total_completed / $total_accessed) * 100) : 0;

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Patient Resources - <?php echo safeHtmlspecialchars($patient['name']); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .page-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem;
            border-radius: 15px;
            margin-bottom: 2rem;
        }
        .stats-card {
            background: white;
            border-radius: 10px;
            padding: 1.5rem;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            text-align: center;
        }
        .status-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 0.75rem;
        }
        .status-completed { background-color: #d4edda; color: #155724; }
        .status-started { background-color: #fff3cd; color: #856404; }
        .status-assigned { background-color: #e2e3e5; color: #383d41; }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="page-header mt-4">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-0">Patient Resource Dashboard</h1>
                            <p class="mb-0">Viewing progress for: <strong><?php echo safeHtmlspecialchars($patient['name']); ?></strong></p>
                        </div>
                        <a href="pcos_education_resources.php" class="btn btn-light">
                            <i class="fas fa-arrow-left me-2"></i>Back to Main Resources
                        </a>
                    </div>
                </div>

                <?php if ($success): ?>
                    <div class="alert alert-success"><?php echo $success; ?></div>
                <?php endif; ?>
                <?php if ($error): ?>
                    <div class="alert alert-danger"><?php echo $error; ?></div>
                <?php endif; ?>

                <!-- Stats -->
                <div class="row mb-4">
                    <div class="col-md-4">
                        <div class="stats-card">
                            <h3 class="text-primary"><?php echo $total_accessed; ?></h3>
                            <p class="text-muted mb-0">Total Accessed/Assigned</p>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="stats-card">
                            <h3 class="text-success"><?php echo $total_completed; ?></h3>
                            <p class="text-muted mb-0">Completed</p>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="stats-card">
                            <h3 class="text-info"><?php echo $completion_rate; ?>%</h3>
                            <p class="text-muted mb-0">Completion Rate</p>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-lg-8">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-history me-2"></i>Resource History</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Resource Title</th>
                                                <th>Category</th>
                                                <th>Status</th>
                                                <th>Date Accessed</th>
                                                <th>Rating</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php if (empty($accessed_resources)): ?>
                                                <tr><td colspan="5" class="text-center text-muted">No resources accessed or assigned yet.</td></tr>
                                            <?php else: ?>
                                                <?php foreach ($accessed_resources as $res): ?>
                                                    <tr>
                                                        <td><?php echo safeHtmlspecialchars($res['title']); ?></td>
                                                        <td><span class="badge bg-secondary"><?php echo ucwords(str_replace('_', ' ', $res['category'])); ?></span></td>
                                                        <td><span class="status-badge status-<?php echo $res['completion_status']; ?>"><?php echo ucfirst($res['completion_status']); ?></span></td>
                                                        <td><?php echo date('M j, Y', strtotime($res['access_date'])); ?></td>
                                                        <td>
                                                            <?php if ($res['rating']): ?>
                                                                <span class="text-warning">
                                                                    <?php for ($i = 0; $i < $res['rating']; $i++) echo '<i class="fas fa-star"></i>'; ?>
                                                                </span>
                                                            <?php else: ?>
                                                                <span class="text-muted">-</span>
                                                            <?php endif; ?>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-user-plus me-2"></i>Assign New Resources</h5>
                            </div>
                            <div class="card-body">
                                <form method="POST">
                                    <div class="mb-3">
                                        <label class="form-label">Available Resources</label>
                                        <div style="max-height: 400px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; border-radius: 5px;">
                                            <?php 
                                            $assigned_ids = array_column($accessed_resources, 'resource_id');
                                            foreach ($all_resources as $resource): 
                                                $is_assigned = in_array($resource['id'], $assigned_ids);
                                            ?>
                                                <div class="form-check">
                                                    <input class="form-check-input" type="checkbox" name="resource_ids[]" value="<?php echo $resource['id']; ?>" id="res_<?php echo $resource['id']; ?>" <?php if ($is_assigned) echo 'disabled'; ?>>
                                                    <label class="form-check-label <?php if ($is_assigned) echo 'text-muted'; ?>" for="res_<?php echo $resource['id']; ?>">
                                                        <?php echo safeHtmlspecialchars($resource['title']); ?>
                                                        <?php if ($is_assigned) echo ' (Assigned)'; ?>
                                                    </label>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                    <button type="submit" name="assign_resources" class="btn btn-primary w-100">
                                        Assign Selected
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>