<?php
require_once '../config.php';
requireRole('doctor');

$db = new Database();
$conn = $db->getConnection();

// Get doctor's ID
$stmt_doc = $conn->prepare("SELECT id FROM doctors WHERE user_id = ?");
$stmt_doc->execute([$_SESSION['user_id']]);
$doctor = $stmt_doc->fetch();
$doctor_id = $doctor['id'] ?? 0;

$patients = [];
$total_records = 0;
$total_pages = 0;
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 15;
$offset = ($page - 1) * $limit;

// --- Filtering Logic ---
$search_term = sanitize($_GET['search'] ?? '');
$risk_level = sanitize($_GET['risk_level'] ?? '');
$age_min = !empty($_GET['age_min']) ? (int)$_GET['age_min'] : null;
$age_max = !empty($_GET['age_max']) ? (int)$_GET['age_max'] : null;

$where_clauses = ["p.doctor_id = ?"];
$params = [$doctor_id];

if ($search_term !== '') {
    $where_clauses[] = "(p.name LIKE ? OR u.email LIKE ?)";
    $params[] = "%$search_term%";
    $params[] = "%$search_term%";
}
if ($risk_level !== '') {
    $where_clauses[] = "latest_assessment.risk_level = ?";
    $params[] = $risk_level;
}
if ($age_min !== null) {
    $where_clauses[] = "p.age >= ?";
    $params[] = $age_min;
}
if ($age_max !== null) {
    $where_clauses[] = "p.age <= ?";
    $params[] = $age_max;
}

$where_sql = 'WHERE ' . implode(' AND ', $where_clauses);

// Base query with a subquery to get the latest assessment for each patient
$base_query = "
    FROM patients p
    JOIN users u ON p.user_id = u.id
    LEFT JOIN (
        SELECT 
            patient_id, risk_level, pcos_probability, created_at,
            ROW_NUMBER() OVER(PARTITION BY patient_id ORDER BY created_at DESC) as rn
        FROM pcos_comprehensive_assessments
    ) AS latest_assessment ON p.id = latest_assessment.patient_id AND latest_assessment.rn = 1
    $where_sql
";

// Get total records for pagination
$count_sql = "SELECT COUNT(DISTINCT p.id) " . $base_query;
$count_stmt = $conn->prepare($count_sql);
$count_stmt->execute($params);
$total_records = $count_stmt->fetchColumn();
$total_pages = ceil($total_records / $limit);

// Fetch patient data
$data_sql = "
    SELECT 
        p.id, p.name, p.age, u.email,
        latest_assessment.risk_level,
        latest_assessment.pcos_probability,
        latest_assessment.created_at as last_assessment_date
    " . $base_query . "
    ORDER BY p.name ASC
    LIMIT $limit OFFSET $offset
";
$stmt_patients = $conn->prepare($data_sql);
$stmt_patients->execute($params);
$patients = $stmt_patients->fetchAll();

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Patient Search - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-search me-2"></i>Advanced Patient Search</h1>
                </div>

                <!-- Search Form -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0">Filter Patients</h5>
                    </div>
                    <div class="card-body">
                        <form method="GET" class="row g-3 align-items-end">
                            <div class="col-md-4">
                                <label for="search" class="form-label">Name or Email</label>
                                <input type="text" class="form-control" id="search" name="search" placeholder="e.g., Jane Doe" value="<?php echo htmlspecialchars($search_term); ?>">
                            </div>
                            <div class="col-md-3">
                                <label for="risk_level" class="form-label">PCOS Risk Level</label>
                                <select class="form-select" id="risk_level" name="risk_level">
                                    <option value="">All Levels</option>
                                    <option value="High" <?php if ($risk_level === 'High') echo 'selected'; ?>>High</option>
                                    <option value="Moderate-High" <?php if ($risk_level === 'Moderate-High') echo 'selected'; ?>>Moderate-High</option>
                                    <option value="Moderate" <?php if ($risk_level === 'Moderate') echo 'selected'; ?>>Moderate</option>
                                    <option value="Low" <?php if ($risk_level === 'Low') echo 'selected'; ?>>Low</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Age Range</label>
                                <div class="input-group">
                                    <input type="number" class="form-control" name="age_min" placeholder="Min" value="<?php echo htmlspecialchars($age_min ?? ''); ?>">
                                    <input type="number" class="form-control" name="age_max" placeholder="Max" value="<?php echo htmlspecialchars($age_max ?? ''); ?>">
                                </div>
                            </div>
                            <div class="col-md-2">
                                <button type="submit" class="btn btn-primary w-100"><i class="fas fa-filter me-2"></i>Search</button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Search Results -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Search Results (<?php echo $total_records; ?> found)</h5>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th>Name</th>
                                        <th>Age</th>
                                        <th>Email</th>
                                        <th>Last Risk Level</th>
                                        <th>Last Assessment</th>
                                        <th class="text-center">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($patients)): ?>
                                        <tr><td colspan="6" class="text-center text-muted p-4">No patients found matching your criteria.</td></tr>
                                    <?php else: ?>
                                        <?php foreach ($patients as $patient): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($patient['name']); ?></td>
                                                <td><?php echo htmlspecialchars($patient['age']); ?></td>
                                                <td><?php echo htmlspecialchars($patient['email']); ?></td>
                                                <td>
                                                    <?php if ($patient['risk_level']): ?>
                                                        <span class="badge bg-<?php echo strtolower($patient['risk_level']) == 'high' ? 'danger' : (strpos(strtolower($patient['risk_level']), 'moderate') !== false ? 'warning' : 'success'); ?>">
                                                            <?php echo htmlspecialchars($patient['risk_level']); ?>
                                                        </span>
                                                    <?php else: ?>
                                                        <span class="badge bg-secondary">Not Assessed</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td><?php echo $patient['last_assessment_date'] ? date('M j, Y', strtotime($patient['last_assessment_date'])) : 'N/A'; ?></td>
                                                <td class="text-center">
                                                    <a href="patient_profile.php?id=<?php echo $patient['id']; ?>" class="btn btn-sm btn-outline-primary" title="View Profile"><i class="fas fa-user"></i></a>
                                                    <a href="add_record.php?patient_id=<?php echo $patient['id']; ?>" class="btn btn-sm btn-outline-info" title="New Record"><i class="fas fa-file-plus"></i></a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <?php if ($total_pages > 1): ?>
                    <div class="card-footer">
                        <nav>
                            <ul class="pagination justify-content-center mb-0">
                                <?php $query_params = $_GET; ?>
                                <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?<?php echo http_build_query(array_merge($query_params, ['page' => $page - 1])); ?>">Previous</a>
                                </li>
                                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                    <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                        <a class="page-link" href="?<?php echo http_build_query(array_merge($query_params, ['page' => $i])); ?>"><?php echo $i; ?></a>
                                    </li>
                                <?php endfor; ?>
                                <li class="page-item <?php echo $page >= $total_pages ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?<?php echo http_build_query(array_merge($query_params, ['page' => $page + 1])); ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>