<?php
require_once '../config.php';
requireRole('doctor'); // Only doctors can view all patients

// The safeHtmlspecialchars function is now in config.php
// function safeHtmlspecialchars($value, $default = '') { return htmlspecialchars($value ?? $default, ENT_QUOTES, 'UTF-8'); }

$db = new Database();
$conn = $db->getConnection();

// Get doctor info
$stmt = $conn->prepare("SELECT * FROM doctors WHERE user_id = ?");
$stmt->execute([$_SESSION['user_id']]);
$doctor = $stmt->fetch();

// Handle new appointment form
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['schedule_appointment'])) {
    $patient_id = (int)$_POST['patient_id'];
    $appointment_date = $_POST['appointment_date'];
    $appointment_time = $_POST['appointment_time'];
    $reason = sanitize($_POST['reason']);
    
    try {
        $stmt = $conn->prepare("INSERT INTO appointments 
            (patient_id, doctor_id, appointment_date, appointment_time, reason, status)
            VALUES (?, ?, ?, ?, ?, 'scheduled')");
        
        $stmt->execute([$patient_id, $doctor['id'], $appointment_date, $appointment_time, $reason]);
        
        $success = 'Appointment scheduled successfully!';
        logActivity($_SESSION['user_id'], 'Scheduled appointment', "Patient ID: $patient_id, Date: $appointment_date");
        
    } catch (Exception $e) {
        $error = 'Failed to schedule appointment: ' . $e->getMessage();
    }
}

// Fetch all registered patients with their profile info for the table
$stmt = $conn->prepare("
    SELECT p.*, u.username, u.email
    FROM patients p
    JOIN users u ON p.user_id = u.id
    WHERE p.doctor_id = ?
    ORDER BY p.name ASC
");
$stmt->execute([$doctor['id']]);
$patients = $stmt->fetchAll();

// Fetch all patients for the modal dropdown
$stmt_modal = $conn->query("SELECT * FROM patients ORDER BY name");
$patients_for_modal = $stmt_modal->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Registered Patients - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body style="background: linear-gradient(to bottom right, #f0f4f8, #e9f0f7);">
    <?php include 'includes/navbar.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <?php if (file_exists('includes/sidebar.php')) include 'includes/sidebar.php'; ?>
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4" style="background-color: transparent;">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Registered Patients</h1>
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addAppointmentModal">
                        <i class="fas fa-plus me-2"></i>Schedule Appointment
                    </button>
                </div>

                <?php if (isset($success)): ?>
                    <div class="alert alert-success alert-dismissible fade show">
                        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if (isset($error)): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">All Patients</h5>
                        <div class="w-50">
                            <div class="input-group">
                                <span class="input-group-text"><i class="fas fa-search"></i></span>
                                <input type="text" id="patientSearch" class="form-control" placeholder="Search patients by name, email, or contact...">
                            </div>
                        </div>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                        <table class="table table-bordered table-hover align-middle">
                            <thead class="table-light">
                                <tr>
                                    <th>#</th>
                                    <th>Name</th>
                                    <th>Username</th>
                                    <th>Email</th>
                                    <th>Age</th>
                                    <th>Gender</th>
                                    <th>Contact</th>
                                    <th>Address</th>
                                    <th>Medical History</th>
                                    <th class="text-center">Actions</th>
                                </tr>
                            </thead>
                            <tbody id="patientTableBody">
                                <?php if ($patients): ?>
                                    <?php foreach ($patients as $i => $p): ?>
                                        <tr>
                                            <td><?php echo $i + 1; ?></td><td><?php echo safeHtmlspecialchars($p['name']); ?></td>
                                            <td><?php echo safeHtmlspecialchars($p['username']); ?></td>
                                            <td><?php echo safeHtmlspecialchars($p['email']); ?></td>
                                            <td><?php echo safeHtmlspecialchars($p['age']); ?></td>
                                            <td><?php echo safeHtmlspecialchars($p['gender']); ?></td>
                                            <td><?php echo safeHtmlspecialchars($p['contact']); ?></td>
                                            <td><?php echo safeHtmlspecialchars($p['address']); ?></td>
                                            <td><?php echo safeHtmlspecialchars($p['medical_history']); ?></td>
                                            <td class="text-center">
                                                <button class="btn btn-sm btn-outline-primary schedule-btn"
                                                        data-bs-toggle="modal"
                                                        data-bs-target="#addAppointmentModal"
                                                        data-patient-id="<?php echo $p['id']; ?>"
                                                        title="Schedule Appointment">
                                                    <i class="fas fa-calendar-plus"></i>
                                                </button>
                                                <a href="view_scans.php?patient_id=<?php echo $p['id']; ?>" 
                                                   class="btn btn-sm btn-outline-info" 
                                                   title="View Uploaded Scans">
                                                    <i class="fas fa-images"></i>
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr id="noResultsRow">
                                        <td colspan="10" class="text-center">No patients found.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- Add Appointment Modal -->
    <div class="modal fade" id="addAppointmentModal" tabindex="-1" aria-labelledby="addAppointmentModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addAppointmentModalLabel">Schedule New Appointment</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="patient_id" class="form-label">Patient *</label>
                            <select class="form-select" id="patient_id" name="patient_id" required>
                                <option value="">Select Patient</option>
                                <?php foreach ($patients_for_modal as $patient): ?>
                                    <option value="<?php echo $patient['id']; ?>">
                                        <?php echo safeHtmlspecialchars($patient['name']) . ' (Age: ' . safeHtmlspecialchars($patient['age']) . ')'; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="appointment_date" class="form-label">Date *</label>
                                <input type="date" class="form-control" id="appointment_date" name="appointment_date" required min="<?php echo date('Y-m-d'); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="appointment_time" class="form-label">Time *</label>
                                <input type="time" class="form-control" id="appointment_time" name="appointment_time" required>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="reason" class="form-label">Reason for Appointment *</label>
                            <textarea class="form-control" id="reason" name="reason" rows="3" required placeholder="Enter the reason for this appointment..."></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="schedule_appointment" class="btn btn-primary">Schedule Appointment</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.getElementById('patientSearch').addEventListener('keyup', function() {
            let searchTerm = this.value.toLowerCase();
            let tableBody = document.getElementById('patientTableBody');
            let rows = tableBody.getElementsByTagName('tr');
            let noResultsRow = document.getElementById('noResultsRow');
            let found = false;

            for (let i = 0; i < rows.length; i++) {
                let row = rows[i];
                if (row.id === 'noResultsRow') continue;

                let rowText = row.textContent.toLowerCase();
                if (rowText.includes(searchTerm)) {
                    row.style.display = '';
                    found = true;
                } else {
                    row.style.display = 'none';
                }
            }

            noResultsRow.style.display = found ? 'none' : '';
        });

        // Handle pre-selecting patient in modal
        var addAppointmentModal = document.getElementById('addAppointmentModal');
        addAppointmentModal.addEventListener('show.bs.modal', function (event) {
            var button = event.relatedTarget;
            var patientId = button.getAttribute('data-patient-id');
            var patientSelect = addAppointmentModal.querySelector('#patient_id');

            if (patientId) {
                patientSelect.value = patientId;
                patientSelect.disabled = true; // Optional: prevent changing the patient
            }
        });
    </script>
</body>
</html>