<?php
require_once '../config.php';
requireRole('doctor');

// The safeHtmlspecialchars function is now in config.php, but let's ensure it's here for clarity if needed.
// function safeHtmlspecialchars($value, $default = '') { return htmlspecialchars($value ?? $default, ENT_QUOTES, 'UTF-8'); }

$db = new Database();
$conn = $db->getConnection();

// Get doctor info
$stmt = $conn->prepare("SELECT * FROM doctors WHERE user_id = ?");
$stmt->execute([$_SESSION['user_id']]);
$doctor = $stmt->fetch();

// Get comprehensive assessment statistics
$stmt = $conn->query("
    SELECT 
        COUNT(*) as total_assessments,
        COUNT(CASE WHEN risk_level = 'High' THEN 1 END) as high_risk,
        COUNT(CASE WHEN risk_level = 'Moderate-High' THEN 1 END) as moderate_high_risk,
        COUNT(CASE WHEN pcos_probability >= 70 THEN 1 END) as likely_pcos,
        AVG(pcos_probability) as avg_probability,
        COUNT(CASE WHEN physician_reviewed = 0 THEN 1 END) as pending_review,
        AVG(pca.bmi) as avg_bmi,
        COUNT(CASE WHEN pca.systolic_bp >= 130 OR pca.diastolic_bp >= 85 THEN 1 END) as high_bp_count,
        COUNT(CASE WHEN pca.ldl_cholesterol >= 130 OR pca.hdl_cholesterol < 50 OR pca.triglycerides >= 150 THEN 1 END) as dyslipidemia_count,
        COUNT(CASE WHEN pca.sleep_quality = 'poor' THEN 1 END) as poor_sleep_count
    FROM pcos_comprehensive_assessments pca
    JOIN patients p ON pca.patient_id = p.id
    WHERE DATE(pca.created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
");
$stats = $stmt->fetch();

// Get recent high-priority assessments needing review
$stmt = $conn->query("
    SELECT 
        pca.*,
        p.name as patient_name,
        p.age,
        p.contact,
        DATEDIFF(CURDATE(), pca.assessment_date) as days_since_assessment
    FROM pcos_comprehensive_assessments pca
    JOIN patients p ON pca.patient_id = p.id
    WHERE pca.physician_reviewed = 0 
       AND (pca.risk_level IN ('High', 'Moderate-High') OR pca.pcos_probability >= 70)
    ORDER BY pca.pcos_probability DESC, pca.created_at ASC
    LIMIT 10
");
$pending_reviews = $stmt->fetchAll();

// Get patients with active recommendations
$stmt = $conn->query("
    SELECT 
        pr.patient_id,
        p.name as patient_name,
        COUNT(CASE WHEN pr.status = 'pending' AND pr.priority_level = 'urgent' THEN 1 END) as urgent_recs,
        COUNT(CASE WHEN pr.status = 'pending' AND pr.priority_level = 'high' THEN 1 END) as high_recs,
        COUNT(CASE WHEN pr.status = 'pending' THEN 1 END) as total_pending,
        MAX(pr.created_at) as latest_recommendation
    FROM pcos_recommendations pr
    JOIN patients p ON pr.patient_id = p.id
    WHERE pr.status IN ('pending', 'in_progress')
    GROUP BY pr.patient_id, p.name
    HAVING total_pending > 0
    ORDER BY urgent_recs DESC, high_recs DESC, latest_recommendation DESC
    LIMIT 10
");
$active_recommendations = $stmt->fetchAll();

// Get real-time assessment data for the dashboard
$stmt = $conn->query("
    SELECT 
        pca.*,
        p.name as patient_name,
        p.age,
        p.contact,
        TIMESTAMPDIFF(MINUTE, pca.created_at, NOW()) as minutes_ago
    FROM pcos_comprehensive_assessments pca
    JOIN patients p ON pca.patient_id = p.id
    WHERE pca.created_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
    ORDER BY pca.created_at DESC
    LIMIT 5
");
$recent_assessments = $stmt->fetchAll();

// Get patients with unread notifications sent by this doctor
$stmt = $conn->prepare("
    SELECT
        p.id as patient_id,
        p.name as patient_name,
        COUNT(n.id) as unread_count,
        MAX(n.created_at) as last_notification_date
    FROM notifications n
    JOIN patients p ON n.user_id = p.user_id
    WHERE n.sender_id = ? AND n.is_read = 0
    GROUP BY p.id, p.name
    ORDER BY last_notification_date DESC
");
$stmt->execute([$doctor['user_id']]);
$unread_notifications_by_patient = $stmt->fetchAll();

// Get recent simple PCOS assessments
$stmt_simple = $conn->query("
    SELECT 
        pa.*,
        p.name as patient_name,
        p.id as patient_id
    FROM pcos_assessments pa
    JOIN patients p ON pa.patient_id = p.id
    ORDER BY pa.created_at DESC
    LIMIT 5
");
$simple_assessments = $stmt_simple->fetchAll();

// Handle physician review submission
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['review_assessment'])) {
    $assessment_id = (int)$_POST['assessment_id'];
    $physician_notes = sanitize($_POST['physician_notes']);
    $approved_recommendations = $_POST['approved_recommendations'] ?? [];
    
    try {
        // Update assessment as reviewed
        $stmt = $conn->prepare("UPDATE pcos_comprehensive_assessments 
                               SET physician_reviewed = 1, physician_notes = ? 
                               WHERE id = ?");
        $stmt->execute([$physician_notes, $assessment_id]);
        
        // Update approved recommendations
        if (!empty($approved_recommendations)) {
            $placeholders = str_repeat('?,', count($approved_recommendations) - 1) . '?';
            $stmt = $conn->prepare("UPDATE pcos_recommendations 
                                   SET physician_approved = 1 
                                   WHERE id IN ($placeholders)");
            $stmt->execute($approved_recommendations);
        }
        
        $success = "Assessment reviewed and recommendations approved successfully.";
        logActivity($_SESSION['user_id'], 'Reviewed PCOS assessment', "Assessment ID: $assessment_id");
        
        // Refresh the page to show updated data
        header("Location: pcos_analysis.php");
        exit();
        
    } catch (Exception $e) {
        $error = "Failed to save review: " . $e->getMessage();
    }
}

// Handle notification sending from modal
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['send_notification'])) {
    $patient_id = (int)$_POST['patient_id'];
    $message = sanitize($_POST['message']);
    $sender_id = $_SESSION['user_id'];

    // Get patient's user_id
    $stmt = $conn->prepare("SELECT user_id FROM patients WHERE id = ?");
    $stmt->execute([$patient_id]);
    $patient_user_id = $stmt->fetchColumn();

    if ($patient_user_id && !empty($message)) {
        createNotification($patient_user_id, 'message', $message, 'patient/dashboard.php', $sender_id);
        $success = "Notification sent successfully to the patient.";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PCOS Management Dashboard - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        .stats-card {
            border-left: 4px solid;
            transition: transform 0.2s, box-shadow 0.2s;
        }
        .stats-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        .stats-card.urgent { border-left-color: #dc3545; }
        .stats-card.high { border-left-color: #fd7e14; }
        .stats-card.moderate { border-left-color: #ffc107; }
        .stats-card.info { border-left-color: #0dcaf0; }
        
        .priority-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
        }
        .priority-urgent { background-color: #dc3545; color: white; }
        .priority-high { background-color: #fd7e14; color: white; }
        .priority-moderate { background-color: #ffc107; color: black; }
        .priority-low { background-color: #6c757d; color: white; }
        
        .risk-indicator {
            width: 12px;
            height: 12px;
            border-radius: 50%;
            display: inline-block;
            margin-right: 8px;
        }
        .risk-high { background-color: #dc3545; }
        .risk-moderate-high { background-color: #fd7e14; }
        .risk-moderate { background-color: #ffc107; }
        .risk-low { background-color: #198754; }
        
        .assessment-card {
            border-left: 4px solid #0d6efd;
            margin-bottom: 1rem;
            transition: all 0.3s;
        }
        .assessment-card:hover {
            transform: translateX(5px);
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        
        .probability-circle {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            color: white;
            font-size: 0.9rem;
        }
        .prob-high { background: linear-gradient(45deg, #dc3545, #fd7e14); }
        .prob-moderate { background: linear-gradient(45deg, #fd7e14, #ffc107); }
        .prob-low { background: linear-gradient(45deg, #198754, #20c997); }
        
        .notification-badge {
            position: absolute;
            top: -5px;
            right: -5px;
            background-color: #dc3545;
            color: white;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.7rem;
        }
        
        .realtime-badge {
            background-color: #28a745;
            color: white;
            padding: 2px 6px;
            border-radius: 10px;
            font-size: 0.7rem;
            margin-left: 5px;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% { opacity: 1; }
            50% { opacity: 0.6; }
            100% { opacity: 1; }
        }
    </style>
</head>
<body style="background: linear-gradient(to bottom right, #f0f4f8, #e9f0f7);">
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4" style="background-color: transparent;">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="fas fa-tachometer-alt text-primary me-2"></i>
                        PCOS Management Dashboard
                        <span class="realtime-badge" id="realtimeIndicator">LIVE</span>
                    </h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="pcos_analytics.php" class="btn btn-outline-primary me-2">
                            <i class="fas fa-chart-bar me-2"></i>Analytics
                        </a>
                        <a href="patient_search.php" class="btn btn-primary">
                            <i class="fas fa-search me-2"></i>Find Patient
                        </a>
                    </div>
                </div>
                
                <?php if (isset($success)): ?>
                    <div class="alert alert-success alert-dismissible fade show">
                        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if (isset($error)): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <!-- Statistics Overview -->
                <div class="row mb-4">
                    <div class="col-lg-3 col-md-6 mb-3">
                        <div class="card stats-card urgent">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted mb-0">Pending Reviews</h6>
                                        <h3 class="fw-bold mb-0 text-danger">
                                            <?php echo $stats['pending_review']; ?>
                                            <?php if ($stats['pending_review'] > 0): ?>
                                                <span class="notification-badge"><?php echo $stats['pending_review']; ?></span>
                                            <?php endif; ?>
                                        </h3>
                                    </div>
                                    <div class="icon-shape bg-danger text-white rounded-circle p-3">
                                        <i class="fas fa-exclamation-triangle"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-3 col-md-6 mb-3">
                        <div class="card stats-card high">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted mb-0">High Risk Cases</h6>
                                        <h3 class="fw-bold mb-0 text-warning"><?php echo $stats['high_risk']; ?></h3>
                                    </div>
                                    <div class="icon-shape bg-warning text-dark rounded-circle p-3">
                                        <i class="fas fa-user-injured"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-3 col-md-6 mb-3">
                        <div class="card stats-card moderate">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted mb-0">Likely PCOS</h6>
                                        <h3 class="fw-bold mb-0 text-info"><?php echo $stats['likely_pcos']; ?></h3>
                                    </div>
                                    <div class="icon-shape bg-info text-white rounded-circle p-3">
                                        <i class="fas fa-microscope"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-3 col-md-6 mb-3">
                        <div class="card stats-card info">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted mb-0">Total Assessments</h6>
                                        <h3 class="fw-bold mb-0 text-primary"><?php echo $stats['total_assessments']; ?></h3>
                                        <small class="text-muted">Last 30 days</small>
                                    </div>
                                    <div class="icon-shape bg-primary text-white rounded-circle p-3">
                                        <i class="fas fa-clipboard-list"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Metabolic & Lifestyle Statistics -->
                <div class="row mb-4">
                    <div class="col-lg-3 col-md-6 mb-3">
                        <div class="card stats-card info">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted mb-0">Avg. BMI</h6>
                                        <h3 class="fw-bold mb-0 text-primary"><?php echo $stats['avg_bmi'] ? round($stats['avg_bmi'], 1) : 'N/A'; ?></h3>
                                        <small class="text-muted">Last 30 days</small>
                                    </div>
                                    <div class="icon-shape bg-primary text-white rounded-circle p-3">
                                        <i class="fas fa-weight"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-3 col-md-6 mb-3">
                        <div class="card stats-card high">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted mb-0">High Blood Pressure</h6>
                                        <h3 class="fw-bold mb-0 text-warning"><?php echo $stats['high_bp_count']; ?></h3>
                                        <small class="text-muted">Last 30 days</small>
                                    </div>
                                    <div class="icon-shape bg-warning text-dark rounded-circle p-3">
                                        <i class="fas fa-heartbeat"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-3 col-md-6 mb-3">
                        <div class="card stats-card high">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted mb-0">Dyslipidemia Cases</h6>
                                        <h3 class="fw-bold mb-0 text-warning"><?php echo $stats['dyslipidemia_count']; ?></h3>
                                        <small class="text-muted">Last 30 days</small>
                                    </div>
                                    <div class="icon-shape bg-warning text-dark rounded-circle p-3">
                                        <i class="fas fa-tint"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-3 col-md-6 mb-3">
                        <div class="card stats-card moderate">
                            <div class="card-body">
                                <h6 class="card-title text-muted mb-0">Poor Sleep Quality</h6>
                                <h3 class="fw-bold mb-0 text-info"><?php echo $stats['poor_sleep_count']; ?></h3>
                                <small class="text-muted">Last 30 days</small>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Real-time Recent Assessments -->
                <?php if (!empty($recent_assessments)): ?>
                <div class="card mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">
                            <i class="fas fa-clock text-primary me-2"></i>
                            Recent Assessments (Last Hour)
                        </h5>
                        <span class="badge bg-success">Real-time</span>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Patient</th>
                                        <th>Time</th>
                                        <th>Probability</th>
                                        <th>Risk Level</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($recent_assessments as $assessment): ?>
                                    <tr>
                                        <td><?php echo safeHtmlspecialchars($assessment['patient_name']); ?></td>
                                        <td>
                                            <?php 
                                            if ($assessment['minutes_ago'] < 1) {
                                                echo "Just now";
                                            } elseif ($assessment['minutes_ago'] < 60) {
                                                echo $assessment['minutes_ago'] . " min ago";
                                            } else {
                                                echo floor($assessment['minutes_ago'] / 60) . " hour(s) ago";
                                            }
                                            ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php 
                                                echo $assessment['pcos_probability'] >= 70 ? 'danger' : 
                                                    ($assessment['pcos_probability'] >= 40 ? 'warning' : 'success');
                                            ?>">
                                                <?php echo $assessment['pcos_probability']; ?>%
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php 
                                                echo $assessment['risk_level'] == 'High' ? 'danger' : 
                                                    ($assessment['risk_level'] == 'Moderate-High' ? 'warning' : 
                                                    ($assessment['risk_level'] == 'Moderate' ? 'info' : 'secondary'));
                                            ?>">
                                                <?php echo $assessment['risk_level']; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($assessment['physician_reviewed'] == 0): ?>
                                                <span class="badge bg-warning">Pending Review</span>
                                            <?php else: ?>
                                                <span class="badge bg-success">Reviewed</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                
                <div class="row">
                    <!-- Pending Reviews Section -->
                    <div class="col-lg-8">
                        <div class="card mb-4">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <h5 class="mb-0">
                                    <i class="fas fa-clipboard-check text-primary me-2"></i>
                                    High Priority Assessments - Pending Review
                                </h5>
                                <?php if (count($pending_reviews) > 0): ?>
                                    <span class="badge bg-danger"><?php echo count($pending_reviews); ?> Pending</span>
                                <?php endif; ?>
                            </div>
                            <div class="card-body">
                                <?php if (empty($pending_reviews)): ?>
                                    <div class="text-center py-4">
                                        <i class="fas fa-check-circle fa-3x text-success mb-3"></i>
                                        <h5>All Caught Up!</h5>
                                        <p class="text-muted">No high-priority assessments pending review.</p>
                                    </div>
                                <?php else: ?>
                                    <?php foreach ($pending_reviews as $assessment): ?>
                                        <div class="assessment-card">
                                            <div class="card-body">
                                                <div class="row align-items-center g-2">
                                                    <div class="col-md-2">
                                                        <div class="probability-circle <?php 
                                                            if ($assessment['pcos_probability'] >= 70) echo 'prob-high';
                                                            elseif ($assessment['pcos_probability'] >= 40) echo 'prob-moderate';
                                                            else echo 'prob-low';
                                                        ?>">
                                                            <?php echo $assessment['pcos_probability']; ?>%
                                                        </div>
                                                    </div>
                                                    <div class="col-md-5">
                                                        <h6 class="mb-1">
                                                            <a href="#" class="text-decoration-none patient-popover"
                                                               data-bs-toggle="popover"
                                                               data-bs-trigger="hover"
                                                               data-bs-placement="right"
                                                               data-bs-html="true"
                                                               data-patient-id="<?php echo $assessment['patient_id']; ?>"
                                                               title="Recent Symptom Trends"><?php echo htmlspecialchars($assessment['patient_name']); ?></a>
                                                        </h6>
                                                        <p class="mb-1">
                                                            <span class="risk-indicator risk-<?php echo strtolower(str_replace('-', '_', $assessment['risk_level'])); ?>"></span>
                                                            <?php echo $assessment['diagnosis_category']; ?>
                                                        </p>
                                                        <small class="text-muted">
                                                            Age: <?php echo $assessment['age']; ?> | 
                                                            Assessed: <?php echo $assessment['days_since_assessment']; ?> days ago |
                                                            Rotterdam Score: <?php echo $assessment['rotterdam_score']; ?>/6
                                                        </small>
                                                    </div>
                                                    <div class="col-md-5 text-md-end mt-2 mt-md-0">
                                                        <button class="btn btn-sm btn-primary me-2" 
                                                                onclick="reviewAssessment(<?php echo $assessment['id']; ?>)"
                                                                data-bs-toggle="modal" 
                                                                data-bs-target="#reviewModal">
                                                            <i class="fas fa-eye me-1"></i>Review
                                                        </button>
                                                        <a href="view_comprehensive_assessment.php?id=<?php echo $assessment['id']; ?>" 
                                                           class="btn btn-sm btn-outline-secondary">
                                                            <i class="fas fa-file-medical me-1"></i>Details
                                                        </a>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Active Recommendations Section -->
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">
                                    <i class="fas fa-tasks text-primary me-2"></i>
                                    Patients with Active Recommendations
                                </h5>
                            </div>
                            <div class="card-body">
                                <?php if (empty($active_recommendations)): ?>
                                    <div class="text-center py-4">
                                        <i class="fas fa-clipboard-check fa-3x text-success mb-3"></i>
                                        <h5>No Active Recommendations</h5>
                                        <p class="text-muted">All patient recommendations are up to date.</p>
                                    </div>
                                <?php else: ?>
                                    <div class="table-responsive">
                                        <table class="table table-hover">
                                            <thead>
                                                <tr>
                                                    <th>Patient</th>
                                                    <th>Urgent</th>
                                                    <th>High Priority</th>
                                                    <th>Total Pending</th>
                                                    <th>Latest</th>
                                                    <th>Actions</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($active_recommendations as $rec): ?>
                                                    <tr><td><?php echo safeHtmlspecialchars($rec['patient_name']); ?></td>
                                                        <td>
                                                            <?php if ($rec['urgent_recs'] > 0): ?>
                                                                <span class="priority-badge priority-urgent">
                                                                    <?php echo $rec['urgent_recs']; ?>
                                                                </span>
                                                            <?php else: ?>
                                                                <span class="text-muted">-</span>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td>
                                                            <?php if ($rec['high_recs'] > 0): ?>
                                                                <span class="priority-badge priority-high">
                                                                    <?php echo $rec['high_recs']; ?>
                                                                </span>
                                                            <?php else: ?>
                                                                <span class="text-muted">-</span>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td>
                                                            <span class="badge bg-secondary"><?php echo $rec['total_pending']; ?></span>
                                                        </td>
                                                        <td>
                                                            <small class="text-muted">
                                                                <?php echo date('M j, Y', strtotime($rec['latest_recommendation'])); ?>
                                                            </small>
                                                        </td>
                                                        <td>
                                                            <a href="patient_recommendations.php?patient_id=<?php echo $rec['patient_id']; ?>" 
                                                               class="btn btn-sm btn-outline-primary">
                                                                <i class="fas fa-eye me-1"></i>View
                                                            </a>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Right Sidebar -->
                    <div class="col-lg-4">
                        <!-- Recent Simple Assessments -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h6 class="mb-0"><i class="fas fa-clipboard-question me-2 text-primary"></i>Recent Simple Assessments</h6>
                            </div>
                            <div class="card-body p-0">
                                <?php if (empty($simple_assessments)): ?>
                                    <p class="text-muted text-center small p-3 mb-0">No recent simple assessments found.</p>
                                <?php else: ?>
                                    <ul class="list-group list-group-flush">
                                        <?php foreach ($simple_assessments as $sa): 
                                            $risk_class = 'secondary';
                                            if ($sa['pcos_risk_level'] == 'High') $risk_class = 'danger';
                                            elseif ($sa['pcos_risk_level'] == 'Moderate') $risk_class = 'warning';
                                            elseif ($sa['pcos_risk_level'] == 'Low') $risk_class = 'info';
                                        ?>
                                            <li class="list-group-item">
                                                <div class="d-flex justify-content-between">
                                                    <a href="patient_profile.php?id=<?php echo $sa['patient_id']; ?>" class="text-decoration-none fw-bold">
                                                        <?php echo safeHtmlspecialchars($sa['patient_name']); ?>
                                                    </a>
                                                    <small class="text-muted"><?php echo date('M j, Y', strtotime($sa['created_at'])); ?></small>
                                                </div>
                                                <div>
                                                    Score: <strong><?php echo $sa['pcos_score']; ?></strong>
                                                    <span class="badge bg-<?php echo $risk_class; ?> float-end">
                                                        <?php echo $sa['pcos_risk_level']; ?> Risk
                                                    </span>
                                                </div>
                                            </li>
                                        <?php endforeach; ?>
                                    </ul>
                                    <div class="card-footer text-center">
                                        <a href="#" class="text-decoration-none small">View All Simple Assessments</a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Quick Actions -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h6 class="mb-0">Quick Actions</h6>
                            </div>
                            <div class="card-body">
                                <div class="d-grid gap-2">
                                    <a href="add_patient.php" class="btn btn-primary">
                                        <i class="fas fa-user-plus me-2"></i>Add New Patient
                                    </a>
                                    <a href="schedule_appointment.php" class="btn btn-outline-primary">
                                        <i class="fas fa-calendar-plus me-2"></i>Schedule Appointment
                                    </a>
                                    <a href="pcos_education_resources.php" class="btn btn-outline-secondary">
                                        <i class="fas fa-book-medical me-2"></i>Patient Resources
                                    </a>
                                    <a href="treatment_guidelines.php" class="btn btn-outline-info">
                                        <i class="fas fa-file-prescription me-2"></i>Treatment Guidelines
                                    </a>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Patient Follow-up (Unread Notifications) -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h6 class="mb-0"><i class="fas fa-comment-dots me-2 text-info"></i>Patient Follow-up</h6>
                            </div>
                            <div class="card-body">
                                <?php if (empty($unread_notifications_by_patient)): ?>
                                    <p class="text-muted text-center small mb-0">All sent notifications have been read.</p>
                                <?php else: ?>
                                    <ul class="list-group list-group-flush">
                                        <?php foreach ($unread_notifications_by_patient as $item): ?>
                                            <li class="list-group-item d-flex justify-content-between align-items-center px-0">
                                                <div>
                                                    <a href="patient_profile.php?id=<?php echo $item['patient_id']; ?>" class="text-decoration-none">
                                                        <?php echo safeHtmlspecialchars($item['patient_name']); ?>
                                                    </a>
                                                    <br>
                                                    <small class="text-muted">
                                                        <?php 
                                                            $date = new DateTime($item['last_notification_date']);
                                                            $now = new DateTime();
                                                            $interval = $now->diff($date);
                                                            if ($interval->d > 0) {
                                                                echo $interval->format('%a days ago');
                                                            } else {
                                                                echo $interval->format('%h hours ago');
                                                            }
                                                        ?>
                                                    </small>
                                                </div>
                                                <span class="badge bg-danger rounded-pill"><?php echo $item['unread_count']; ?> unread</span>
                                            </li>
                                        <?php endforeach; ?>
                                    </ul>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Recent Activity -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h6 class="mb-0">Recent Activity</h6>
                            </div>
                            <div class="card-body">
                                <?php
                                // Get recent activity
                                $stmt = $conn->prepare("
                                    SELECT al.*, u.username 
                                    FROM activity_logs al 
                                    JOIN users u ON al.user_id = u.id 
                                    WHERE al.user_id = ? 
                                    ORDER BY al.timestamp DESC 
                                    LIMIT 5
                                ");
                                $stmt->execute([$_SESSION['user_id']]);
                                $activities = $stmt->fetchAll();
                                ?>
                                
                                <?php if (empty($activities)): ?>
                                    <p class="text-muted">No recent activity.</p>
                                <?php else: ?>
                                    <div class="timeline">
                                        <?php foreach ($activities as $activity): ?>
                                            <div class="timeline-item mb-3">
                                                <small class="text-muted">
                                                    <?php echo date('M j, g:i A', strtotime($activity['timestamp'])); ?>
                                                </small><p class="mb-0"><?php echo safeHtmlspecialchars($activity['action']); ?></p>
                                                <?php if ($activity['details']): ?>
                                                    <small class="text-muted"><?php echo safeHtmlspecialchars($activity['details']); ?></small>
                                                <?php endif; ?>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- PCOS Guidelines Summary -->
                        <div class="card">
                            <div class="card-header">
                                <h6 class="mb-0">PCOS Management Guidelines</h6>
                            </div>
                            <div class="card-body">
                                <h6 class="text-primary">Rotterdam Criteria (2 of 3 required):</h6>
                                <ul class="small">
                                    <li>Oligo/anovulation</li>
                                    <li>Clinical/biochemical hyperandrogenism</li>
                                    <li>Polycystic ovarian morphology (≥20 follicles)</li>
                                </ul>
                                
                                <h6 class="text-primary mt-3">First-line Treatments:</h6>
                                <ul class="small">
                                    <li>Lifestyle modification (weight loss if BMI >25)</li>
                                    <li>Metformin for metabolic features</li>
                                    <li>OCPs for menstrual regulation</li>
                                    <li>Anti-androgens for hirsutism</li>
                                </ul>
                                
                                <h6 class="text-primary mt-3">Follow-up Schedule:</h6>
                                <ul class="small">
                                    <li>High risk: Monthly x3, then quarterly</li>
                                    <li>Moderate risk: Every 3 months x1 year</li>
                                    <li>Low risk: Annual assessment</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- Notify Patient Modal -->
    <div class="modal fade" id="notifyPatientModal" tabindex="-1" aria-labelledby="notifyPatientModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="notifyPatientModalLabel">Send Notification</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" id="notify_patient_id" name="patient_id">
                        <div class="mb-3">
                            <label class="form-label">To:</label>
                            <input type="text" id="notify_patient_name" class="form-control" readonly>
                        </div>
                        <div class="mb-3">
                            <label for="notification_message" class="form-label">Message</label>
                            <textarea class="form-control" id="notification_message" name="message" rows="4" required placeholder="Type your message to the patient..."></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="send_notification" class="btn btn-primary">
                            <i class="fas fa-paper-plane me-2"></i>Send Notification
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Assessment Review Modal -->
    <div class="modal fade" id="reviewModal" tabindex="-1" aria-labelledby="reviewModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="reviewModalLabel">Review PCOS Assessment</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div id="assessmentDetails">
                            <!-- Assessment details will be loaded here -->
                        </div>
                        
                        <div class="mt-4">
                            <label for="physician_notes" class="form-label">Physician Notes</label>
                            <textarea class="form-control" id="physician_notes" name="physician_notes" rows="4" 
                                      placeholder="Add your clinical notes and assessment..."></textarea>
                        </div>
                        
                        <div class="mt-3">
                            <h6>Approve Recommendations:</h6>
                            <div id="recommendationsList">
                                <!-- Recommendations will be loaded here -->
                            </div>
                        </div>
                        
                        <input type="hidden" id="assessment_id" name="assessment_id">
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="review_assessment" class="btn btn-primary">Save Review</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Initialize Bootstrap Popovers
        var popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
        var popoverList = popoverTriggerList.map(function (popoverTriggerEl) {
            return new bootstrap.Popover(popoverTriggerEl, {
                content: 'Loading...'
            });
        });

        // Handle popover content loading via AJAX
        document.querySelectorAll('.patient-popover').forEach(popoverEl => {
            popoverEl.addEventListener('show.bs.popover', function () {
                const popover = bootstrap.Popover.getInstance(this);
                const patientId = this.dataset.patientId;

                // Only fetch if content is the default 'Loading...'
                if (popover && popover._config.content === 'Loading...') {
                    fetch(`get_patient_symptom_trends.php?patient_id=${patientId}`)
                        .then(response => response.json())
                        .then(data => {
                            popover.setContent({ '.popover-body': data.content });
                        })
                        .catch(error => {
                            console.error('Error fetching symptom trends:', error);
                            popover.setContent({ '.popover-body': 'Could not load data.' });
                        });
                }
            });
        });


        // Handle Notify Patient Modal
        var notifyModal = document.getElementById('notifyPatientModal');
        notifyModal.addEventListener('show.bs.modal', function (event) {
            var button = event.relatedTarget;
            var patientId = button.getAttribute('data-patient-id');
            var patientName = button.getAttribute('data-patient-name');

            var modalTitle = notifyModal.querySelector('.modal-title');
            var patientNameInput = notifyModal.querySelector('#notify_patient_name');
            var patientIdInput = notifyModal.querySelector('#notify_patient_id');
            patientNameInput.value = patientName;
            patientIdInput.value = patientId;
        });

        function reviewAssessment(assessmentId) {
            document.getElementById('assessment_id').value = assessmentId;
            
            // Fetch assessment details via AJAX
            fetch(`get_assessment_details.php?id=${assessmentId}`)
                .then(response => response.json())
                .then(data => {
                    document.getElementById('assessmentDetails').innerHTML = data.html;
                    document.getElementById('recommendationsList').innerHTML = data.recommendations;
                })
                .catch(error => {
                    console.error('Error:', error);
                    document.getElementById('assessmentDetails').innerHTML = 
                        '<div class="alert alert-danger">Failed to load assessment details.</div>';
                });
        }
        
        // Auto-refresh pending count every 30 seconds
        setInterval(function() {
            fetch('get_pending_count.php')
                .then(response => response.json())
                .then(data => {
                    // Update notification badges
                    const pendingElement = document.querySelector('.stats-card.urgent .fw-bold');
                    if (pendingElement) {
                        pendingElement.innerHTML = data.pending_count;
                        if (data.pending_count > 0) {
                            pendingElement.innerHTML += `<span class="notification-badge">${data.pending_count}</span>`;
                        }
                    }
                });
        }, 30000);
        
        // Auto-refresh data every 30 seconds
        let refreshInterval = 30000; // 30 seconds
        
        function refreshDashboardData() {
            fetch('get_dashboard_data.php')
                .then(response => response.json())
                .then(data => {
                    // Update statistics
                    if (data.stats) {
                        document.querySelector('.stats-card.urgent .fw-bold').innerHTML = data.stats.pending_review ?? '0';
                        document.querySelector('.stats-card.high .fw-bold').innerHTML = data.stats.high_risk ?? '0';
                        document.querySelector('.stats-card.moderate .fw-bold').innerHTML = data.stats.likely_pcos ?? '0';
                        document.querySelector('.stats-card.info .fw-bold').innerHTML = data.stats.total_assessments ?? '0';
                    }
                    
                    // Update real-time indicator
                    const indicator = document.getElementById('realtimeIndicator');
                    indicator.classList.add('bg-success');
                    setTimeout(() => {
                        indicator.classList.remove('bg-success');
                    }, 2000);
                })
                .catch(error => {
                    console.error('Error refreshing dashboard data:', error);
                });
        }
        
        // Start auto-refresh
        setInterval(refreshDashboardData, refreshInterval);
        
        // Initial data load
        document.addEventListener('DOMContentLoaded', function() {
            refreshDashboardData();
        });
    </script>
</body>
</html>