<?php
require_once '../config.php';
requireRole('doctor');

$db = new Database();
$conn = $db->getConnection();

// Get filter values
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

// Build WHERE clause
$where_conditions = [];
$params = [];

if (!empty($date_from)) {
    $where_conditions[] = "created_at >= ?";
    $params[] = $date_from . ' 00:00:00';
}
if (!empty($date_to)) {
    $where_conditions[] = "created_at <= ?";
    $params[] = $date_to . ' 23:59:59';
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Handle CSV export
if (isset($_GET['export']) && $_GET['export'] == 'csv') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="pcos_analytics_'.date('Y-m-d').'.csv"');

    $output = fopen('php://output', 'w');

    // Add headers to CSV
    fputcsv($output, [
        'Month', 'Total Assessments', 'Avg PCOS Probability (%)', 'Avg BMI', 
        'High Risk Count', 'Moderate-High Risk Count', 'Moderate Risk Count', 'Low Risk Count',
        'High BP Count', 'Dyslipidemia Count', 'Acne Count', 'Hirsutism Count', 
        'Alopecia Count', 'Mood Changes Count'
    ]);

    // Fetch data for CSV
    $sql = "
        SELECT
            DATE_FORMAT(created_at, '%Y-%m') as month, COUNT(*) as total_assessments, AVG(pcos_probability) as avg_probability, AVG(bmi) as avg_bmi,
            SUM(CASE WHEN risk_level = 'High' THEN 1 ELSE 0 END) as high_risk_count, SUM(CASE WHEN risk_level = 'Moderate-High' THEN 1 ELSE 0 END) as moderate_high_risk_count, SUM(CASE WHEN risk_level = 'Moderate' THEN 1 ELSE 0 END) as moderate_risk_count, SUM(CASE WHEN risk_level = 'Low' THEN 1 ELSE 0 END) as low_risk_count,
            SUM(CASE WHEN systolic_bp >= 130 OR diastolic_bp >= 85 THEN 1 ELSE 0 END) as high_bp_count, SUM(CASE WHEN ldl_cholesterol >= 130 OR hdl_cholesterol < 50 OR triglycerides >= 150 THEN 1 ELSE 0 END) as dyslipidemia_count,
            SUM(CASE WHEN acne_severity IN ('moderate', 'severe') THEN 1 ELSE 0 END) as acne_count, SUM(CASE WHEN hirsutism_score >= 8 THEN 1 ELSE 0 END) as hirsutism_count,
            SUM(CASE WHEN androgenic_alopecia = 'yes' THEN 1 ELSE 0 END) as alopecia_count, SUM(CASE WHEN mood_changes = 'yes' THEN 1 ELSE 0 END) as mood_changes_count
        FROM pcos_comprehensive_assessments
        $where_clause
        GROUP BY month ORDER BY month ASC ";
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        fputcsv($output, $row);
    }

    fclose($output);
    exit();
}

// Build WHERE clause
$where_conditions = [];
$params = [];

if (!empty($date_from)) {
    $where_conditions[] = "created_at >= ?";
    $params[] = $date_from . ' 00:00:00';
}
if (!empty($date_to)) {
    $where_conditions[] = "created_at <= ?";
    $params[] = $date_to . ' 23:59:59';
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Fetch monthly analytics data
$sql = "
    SELECT
        DATE_FORMAT(created_at, '%Y-%m') as month,
        COUNT(*) as total_assessments,
        AVG(pcos_probability) as avg_probability,
        AVG(bmi) as avg_bmi,
        SUM(CASE WHEN risk_level = 'High' THEN 1 ELSE 0 END) as high_risk_count, SUM(CASE WHEN risk_level = 'Moderate-High' THEN 1 ELSE 0 END) as moderate_high_risk_count, SUM(CASE WHEN risk_level = 'Moderate' THEN 1 ELSE 0 END) as moderate_risk_count, SUM(CASE WHEN risk_level = 'Low' THEN 1 ELSE 0 END) as low_risk_count,
        SUM(CASE WHEN systolic_bp >= 130 OR diastolic_bp >= 85 THEN 1 ELSE 0 END) as high_bp_count,
        SUM(CASE WHEN ldl_cholesterol >= 130 OR hdl_cholesterol < 50 OR triglycerides >= 150 THEN 1 ELSE 0 END) as dyslipidemia_count,
        SUM(CASE WHEN acne_severity IN ('moderate', 'severe') THEN 1 ELSE 0 END) as acne_count,
        SUM(CASE WHEN hirsutism_score >= 8 THEN 1 ELSE 0 END) as hirsutism_count,
        SUM(CASE WHEN androgenic_alopecia = 'yes' THEN 1 ELSE 0 END) as alopecia_count,
        SUM(CASE WHEN mood_changes = 'yes' THEN 1 ELSE 0 END) as mood_changes_count
    FROM pcos_comprehensive_assessments
    $where_clause
    GROUP BY month ORDER BY month ASC ";
$stmt = $conn->prepare($sql);
$stmt->execute($params);
$monthly_data = $stmt->fetchAll();

// Prepare data for Chart.js
$chart_labels = [];
$assessments_per_month = [];
$avg_probability_per_month = [];
$avg_bmi_per_month = [];
$risk_distribution = [
    'High' => [],
    'Moderate-High' => [],
    'Moderate' => [],
    'Low' => []
];
$metabolic_trends = [
    'high_bp' => [],
    'dyslipidemia' => [],
];
$symptom_trends = [
    'acne' => [],
    'hirsutism' => [],
    'alopecia' => [],
    'mood_changes' => [],
];

foreach ($monthly_data as $data) {
    $chart_labels[] = date('M Y', strtotime($data['month'] . '-01'));
    $assessments_per_month[] = $data['total_assessments'];
    $avg_probability_per_month[] = round($data['avg_probability'], 1);
    $avg_bmi_per_month[] = round($data['avg_bmi'], 1);
    
    $risk_distribution['High'][] = $data['high_risk_count'];
    $risk_distribution['Moderate-High'][] = $data['moderate_high_risk_count'];
    $risk_distribution['Moderate'][] = $data['moderate_risk_count'];
    $risk_distribution['Low'][] = $data['low_risk_count'];

    $metabolic_trends['high_bp'][] = $data['total_assessments'] > 0 ? round(($data['high_bp_count'] / $data['total_assessments']) * 100, 1) : 0;
    $metabolic_trends['dyslipidemia'][] = $data['total_assessments'] > 0 ? round(($data['dyslipidemia_count'] / $data['total_assessments']) * 100, 1) : 0;

    $symptom_trends['acne'][] = $data['total_assessments'] > 0 ? round(($data['acne_count'] / $data['total_assessments']) * 100, 1) : 0;
    $symptom_trends['hirsutism'][] = $data['total_assessments'] > 0 ? round(($data['hirsutism_count'] / $data['total_assessments']) * 100, 1) : 0;
    $symptom_trends['alopecia'][] = $data['total_assessments'] > 0 ? round(($data['alopecia_count'] / $data['total_assessments']) * 100, 1) : 0;
    $symptom_trends['mood_changes'][] = $data['total_assessments'] > 0 ? round(($data['mood_changes_count'] / $data['total_assessments']) * 100, 1) : 0;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PCOS Analytics - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
    <style>
        .chart-container {
            background: white;
            padding: 1.5rem;
            border-radius: 15px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.05);
            margin-bottom: 2rem;
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-chart-line text-primary me-2"></i>PCOS Analytics</h1>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h6 class="mb-0"><i class="fas fa-filter me-2"></i>Filter Analytics</h6>
                    </div>
                    <div class="card-body">
                        <form method="GET" class="row g-3 align-items-end">
                            <div class="col-md-4">
                                <label for="date_from" class="form-label">From</label>
                                <input type="date" class="form-control" id="date_from" name="date_from" value="<?php echo htmlspecialchars($date_from); ?>">
                            </div>
                            <div class="col-md-4">
                                <label for="date_to" class="form-label">To</label>
                                <input type="date" class="form-control" id="date_to" name="date_to" value="<?php echo htmlspecialchars($date_to); ?>">
                            </div>
                            <div class="col-md-4">
                                <div class="d-flex gap-2">
                                    <button type="submit" class="btn btn-primary"><i class="fas fa-search me-1"></i>Apply</button>
                                    <a href="pcos_analytics.php" class="btn btn-outline-secondary"><i class="fas fa-times me-1"></i>Clear</a>
                                    <a href="?<?php echo http_build_query(array_merge($_GET, ['export' => 'csv'])); ?>" class="btn btn-outline-success"><i class="fas fa-file-csv me-1"></i>CSV</a>
                                    <button type="button" id="exportPdfBtn" class="btn btn-outline-danger"><i class="fas fa-file-pdf me-1"></i>PDF Report</button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>


                <?php if (empty($monthly_data)): ?>
                    <div class="alert alert-info text-center">
                        <i class="fas fa-info-circle fa-2x mb-3"></i>
                        <h4>Not Enough Data</h4>
                        <p>There is not enough assessment data to generate analytics. Please add more comprehensive assessments.</p>
                    </div>
                <?php else: ?>
                    <div class="row">
                        <div class="col-lg-7 mb-4">
                            <div class="chart-container">
                                <div class="d-flex justify-content-between align-items-center mb-2">
                                    <h5>Assessments Over Time</h5>
                                    <button class="btn btn-sm btn-outline-secondary export-chart-btn" data-chart-id="assessmentsChart" data-chart-name="assessments-over-time">
                                        <i class="fas fa-download me-1"></i> PNG
                                    </button>
                                </div>
                                <canvas id="assessmentsChart"></canvas>
                            </div>
                        </div>
                        <div class="col-lg-5 mb-4">
                            <div class="chart-container">
                                <div class="d-flex justify-content-between align-items-center mb-2">
                                    <h5>Risk Level Distribution</h5>
                                    <button class="btn btn-sm btn-outline-secondary export-chart-btn" data-chart-id="riskChart" data-chart-name="risk-level-distribution">
                                        <i class="fas fa-download me-1"></i> PNG
                                    </button>
                                </div>
                                <canvas id="riskChart"></canvas>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-lg-6 mb-4">
                            <div class="chart-container">
                                <div class="d-flex justify-content-between align-items-center mb-2">
                                    <h5>Average BMI & PCOS Probability</h5>
                                    <button class="btn btn-sm btn-outline-secondary export-chart-btn" data-chart-id="bmiProbabilityChart" data-chart-name="bmi-probability-trends">
                                        <i class="fas fa-download me-1"></i> PNG
                                    </button>
                                </div>
                                <canvas id="bmiProbabilityChart"></canvas>
                            </div>
                        </div>
                        <div class="col-lg-6 mb-4">
                            <div class="chart-container">
                                <div class="d-flex justify-content-between align-items-center mb-2">
                                    <h5>Metabolic Issues Prevalence (%)</h5>
                                    <button class="btn btn-sm btn-outline-secondary export-chart-btn" data-chart-id="metabolicChart" data-chart-name="metabolic-issues-prevalence">
                                        <i class="fas fa-download me-1"></i> PNG
                                    </button>
                                </div>
                                <canvas id="metabolicChart"></canvas>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-lg-12 mb-4">
                            <div class="chart-container">
                                <div class="d-flex justify-content-between align-items-center mb-2">
                                    <h5>Symptom Prevalence Over Time (%)</h5>
                                    <button class="btn btn-sm btn-outline-secondary export-chart-btn" data-chart-id="symptomsChart" data-chart-name="symptom-prevalence">
                                        <i class="fas fa-download me-1"></i> PNG
                                    </button>
                                </div>
                                <canvas id="symptomsChart"></canvas>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    <?php if (!empty($monthly_data)): ?>
    document.addEventListener('DOMContentLoaded', function() {
        const chartLabels = <?php echo json_encode($chart_labels); ?>;

        // Assessments Chart (Bar)
        const assessmentsCtx = document.getElementById('assessmentsChart').getContext('2d');
        new Chart(assessmentsCtx, {
            type: 'bar',
            data: {
                labels: chartLabels,
                datasets: [{
                    label: 'Total Assessments',
                    data: <?php echo json_encode($assessments_per_month); ?>,
                    backgroundColor: 'rgba(102, 126, 234, 0.6)',
                    borderColor: 'rgba(102, 126, 234, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                scales: { y: { beginAtZero: true } }
            }
        });

        // Risk Distribution Chart (Stacked Bar)
        const riskCtx = document.getElementById('riskChart').getContext('2d');
        new Chart(riskCtx, {
            type: 'bar',
            data: {
                labels: chartLabels,
                datasets: [
                    { label: 'Low', data: <?php echo json_encode($risk_distribution['Low']); ?>, backgroundColor: '#28a745' },
                    { label: 'Moderate', data: <?php echo json_encode($risk_distribution['Moderate']); ?>, backgroundColor: '#ffc107' },
                    { label: 'Moderate-High', data: <?php echo json_encode($risk_distribution['Moderate-High']); ?>, backgroundColor: '#fd7e14' },
                    { label: 'High', data: <?php echo json_encode($risk_distribution['High']); ?>, backgroundColor: '#dc3545' }
                ]
            },
            options: {
                responsive: true,
                scales: { x: { stacked: true }, y: { stacked: true, beginAtZero: true } },
                plugins: { legend: { position: 'bottom' } }
            }
        });

        // BMI & Probability Chart (Line)
        const bmiProbCtx = document.getElementById('bmiProbabilityChart').getContext('2d');
        new Chart(bmiProbCtx, {
            type: 'line',
            data: {
                labels: chartLabels,
                datasets: [
                    {
                        label: 'Avg. BMI',
                        data: <?php echo json_encode($avg_bmi_per_month); ?>,
                        borderColor: '#17a2b8',
                        yAxisID: 'yBmi',
                        tension: 0.3
                    },
                    {
                        label: 'Avg. PCOS Probability (%)',
                        data: <?php echo json_encode($avg_probability_per_month); ?>,
                        borderColor: '#6f42c1',
                        yAxisID: 'yProb',
                        tension: 0.3
                    }
                ]
            },
            options: {
                responsive: true,
                interaction: { mode: 'index', intersect: false },
                scales: {
                    yBmi: {
                        type: 'linear',
                        display: true,
                        position: 'left',
                        title: { display: true, text: 'BMI' }
                    },
                    yProb: {
                        type: 'linear',
                        display: true,
                        position: 'right',
                        min: 0,
                        max: 100,
                        title: { display: true, text: 'Probability (%)' },
                        grid: { drawOnChartArea: false }
                    }
                }
            }
        });

        // Metabolic Issues Chart (Line)
        const metabolicCtx = document.getElementById('metabolicChart').getContext('2d');
        new Chart(metabolicCtx, {
            type: 'line',
            data: {
                labels: chartLabels,
                datasets: [
                    {
                        label: '% with High BP',
                        data: <?php echo json_encode($metabolic_trends['high_bp']); ?>,
                        borderColor: '#dc3545',
                        tension: 0.3,
                        fill: false
                    },
                    {
                        label: '% with Dyslipidemia',
                        data: <?php echo json_encode($metabolic_trends['dyslipidemia']); ?>,
                        borderColor: '#fd7e14',
                        tension: 0.3,
                        fill: false
                    }
                ]
            },
            options: {
                responsive: true,
                scales: {
                    y: {
                        beginAtZero: true,
                        max: 100,
                        ticks: {
                            callback: function(value) { return value + '%' }
                        }
                    }
                }
            }
        });

        // Symptoms Chart (Line)
        const symptomsCtx = document.getElementById('symptomsChart').getContext('2d');
        new Chart(symptomsCtx, {
            type: 'line',
            data: {
                labels: chartLabels,
                datasets: [
                    {
                        label: '% with Acne',
                        data: <?php echo json_encode($symptom_trends['acne']); ?>,
                        borderColor: '#ff6384',
                        tension: 0.3,
                        fill: false
                    },
                    {
                        label: '% with Hirsutism',
                        data: <?php echo json_encode($symptom_trends['hirsutism']); ?>,
                        borderColor: '#36a2eb',
                        tension: 0.3,
                        fill: false
                    },
                    {
                        label: '% with Alopecia',
                        data: <?php echo json_encode($symptom_trends['alopecia']); ?>,
                        borderColor: '#ffce56',
                        tension: 0.3,
                        fill: false
                    },
                    {
                        label: '% with Mood Changes',
                        data: <?php echo json_encode($symptom_trends['mood_changes']); ?>,
                        borderColor: '#4bc0c0',
                        tension: 0.3,
                        fill: false
                    }
                ]
            },
            options: {
                responsive: true,
                scales: { y: { beginAtZero: true, max: 100, ticks: { callback: function(value) { return value + '%' } } } }
            }
        });

        // Add event listeners for chart export buttons
        document.querySelectorAll('.export-chart-btn').forEach(button => {
            button.addEventListener('click', function() {
                const chartId = this.dataset.chartId;
                const chartName = this.dataset.chartName;
                const chart = Chart.getChart(chartId);

                if (chart) {
                    const url = chart.toBase64Image();
                    const link = document.createElement('a');
                    link.href = url;
                    link.download = `${chartName}.png`;
                    document.body.appendChild(link);
                    link.click();
                    document.body.removeChild(link);
                }
            });
        });

        // PDF Export functionality
        document.getElementById('exportPdfBtn').addEventListener('click', async function() {
            const { jsPDF } = window.jspdf;
            const btn = this;
            const originalHtml = btn.innerHTML;
            btn.innerHTML = `<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Generating...`;
            btn.disabled = true;

            const pdf = new jsPDF({
                orientation: 'p',
                unit: 'mm',
                format: 'a4'
            });

            const chartContainers = document.querySelectorAll('.chart-container');
            const pdfWidth = pdf.internal.pageSize.getWidth();
            const pdfHeight = pdf.internal.pageSize.getHeight();
            const margin = 10;

            for (let i = 0; i < chartContainers.length; i++) {
                const container = chartContainers[i];
                const canvas = await html2canvas(container, { scale: 2 });
                const imgData = canvas.toDataURL('image/png');

                if (i > 0) {
                    pdf.addPage();
                }

                const imgProps = pdf.getImageProperties(imgData);
                const imgHeight = (imgProps.height * (pdfWidth - margin * 2)) / imgProps.width;
                pdf.addImage(imgData, 'PNG', margin, margin, pdfWidth - margin * 2, imgHeight);
            }

            pdf.save('pcos_analytics_report_<?php echo date("Y-m-d"); ?>.pdf');
            btn.innerHTML = originalHtml;
            btn.disabled = false;
        });
    });
    <?php endif; ?>
    </script>
</body>
</html>