<?php
require_once '../config.php';
requireRole('doctor');

$db = new Database();
$conn = $db->getConnection();
$success = '';
$error = '';

// Get doctor info for created_by field
$stmt_doc = $conn->prepare("SELECT id FROM doctors WHERE user_id = ?");
$stmt_doc->execute([$_SESSION['user_id']]);
$doctor = $stmt_doc->fetch();

// Handle CSV export

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $upload_dir = '../uploads/resources/';
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0777, true);
    }

    if (isset($_POST['add_resource'])) {
        $title = sanitize($_POST['title']);
        $content = sanitize($_POST['content']); // Changed from description to content
        $category = sanitize($_POST['category']);
        $resource_type = sanitize($_POST['resource_type']);
        $is_active = isset($_POST['is_active']) ? 1 : 0;
        $file_path = '';

        if (isset($_FILES['resource_file']) && $_FILES['resource_file']['error'] == 0) {
            $file = $_FILES['resource_file'];
            $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $new_filename = uniqid('resource_', true) . '.' . $file_extension;
            $upload_path = $upload_dir . $new_filename;

            if (move_uploaded_file($file['tmp_name'], $upload_path)) {
                $file_path = 'uploads/resources/' . $new_filename;

                $stmt = $conn->prepare("
                    INSERT INTO pcos_education_resources 
                    (title, content, category, resource_type, file_path, is_active, created_by)
                    VALUES (?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([$title, $content, $category, $resource_type, $file_path, $is_active, $doctor['id']]);
                $success = "Resource added successfully!";
                logActivity($_SESSION['user_id'], 'Added educational resource', "Title: $title");
            } else {
                $error = "Failed to move uploaded file.";
            }
        } else {
            $error = "A file is required to create a resource.";
        }
    }
    
    if (isset($_POST['update_resource'])) {
        $resource_id = (int)$_POST['resource_id'];
        $title = sanitize($_POST['title']);
        $content = sanitize($_POST['content']); // Changed from description to content
        $category = sanitize($_POST['category']);
        $resource_type = sanitize($_POST['resource_type']);
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        // Fetch old file path for potential deletion (fixed to use file_path instead of content)
        $stmt_old = $conn->prepare("SELECT file_path FROM pcos_education_resources WHERE id = ?");
        $stmt_old->execute([$resource_id]);
        $old_file_path = $stmt_old->fetchColumn();
        $file_path = $old_file_path;

        // Handle new file upload
        if (isset($_FILES['resource_file']) && $_FILES['resource_file']['error'] == 0) {
            $file = $_FILES['resource_file'];
            $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $new_filename = uniqid('resource_', true) . '.' . $file_extension;
            $upload_path = $upload_dir . $new_filename;

            if (move_uploaded_file($file['tmp_name'], $upload_path)) {
                $file_path = 'uploads/resources/' . $new_filename;
                // Delete old file if it exists and is different
                if ($old_file_path && file_exists('../' . $old_file_path) && $old_file_path !== $file_path) {
                    unlink('../' . $old_file_path);
                }
            } else {
                $error = "Failed to move new uploaded file.";
            }
        }

        if (empty($error)) {
            $stmt = $conn->prepare("
                UPDATE pcos_education_resources 
                SET title = ?, content = ?, category = ?, resource_type = ?, file_path = ?, is_active = ?
                WHERE id = ?
            ");
            $stmt->execute([$title, $content, $category, $resource_type, $file_path, $is_active, $resource_id]);
            $success = "Resource updated successfully!";
            logActivity($_SESSION['user_id'], 'Updated educational resource', "ID: $resource_id");
        }
    }
    
    if (isset($_POST['delete_resource'])) {
        $resource_id = (int)$_POST['resource_id'];

        // First, get the file path to delete the file from the server
        $stmt_file = $conn->prepare("SELECT file_path FROM pcos_education_resources WHERE id = ?");
        $stmt_file->execute([$resource_id]);
        $file_to_delete = $stmt_file->fetchColumn();

        // Then, delete the record from the database
        $stmt = $conn->prepare("DELETE FROM pcos_education_resources WHERE id = ?");
        $stmt->execute([$resource_id]);

        if ($stmt->rowCount() > 0) {
            // If DB deletion is successful, delete the file
            if ($file_to_delete && file_exists('../' . $file_to_delete)) {
                unlink('../' . $file_to_delete);
            }
            $success = "Resource deleted successfully!";
            logActivity($_SESSION['user_id'], 'Deleted educational resource', "ID: $resource_id");
        } else {
            $error = "Failed to delete resource.";
        }
    }
    
    if (isset($_POST['assign_resource'])) {
        // Assign resource to specific patients
        $resource_id = (int)$_POST['resource_id'];
        $patient_ids = $_POST['patient_ids'] ?? [];
        
    }
}

// Get all resources
$stmt = $conn->query("
    SELECT * FROM pcos_education_resources 
    ORDER BY created_at DESC
");
$resources = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Resources Management - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .status-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 0.75rem;
        }
        .status-active { background-color: #d4edda; color: #155724; }
        .status-inactive { background-color: #f8d7da; color: #721c24; }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="fas fa-book-medical me-2" style="color: #764ba2;"></i>
                        Educational Resources
                    </h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addResourceModal">
                            <i class="fas fa-plus me-2"></i>Add New Resource
                        </button>
                    </div>
                </div>
                <?php if (isset($success)): ?>
                    <div class="alert alert-success alert-dismissible fade show">
                        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                <?php if (isset($error)): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Uploaded Resources</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Title</th>
                                        <th>Category</th>
                                        <th>Type</th>
                                        <th>Status</th>
                                        <th>Date Added</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($resources)): ?>
                                        <tr><td colspan="6" class="text-center text-muted">No resources uploaded yet.</td></tr>
                                    <?php else: ?>
                                        <?php foreach ($resources as $resource): ?>
                                            <tr>
                                                <td><?php echo safeHtmlspecialchars($resource['title']); ?></td>
                                                <td><span class="badge bg-secondary"><?php echo safeHtmlspecialchars($resource['category']); ?></span></td>
                                                <td><span class="badge bg-info"><?php echo safeHtmlspecialchars($resource['resource_type']); ?></span></td>
                                                <td>
                                                    <span class="status-badge status-<?php echo $resource['is_active'] ? 'active' : 'inactive'; ?>">
                                                        <?php echo $resource['is_active'] ? 'Active' : 'Inactive'; ?>
                                                    </span>
                                                </td>
                                                <td><?php echo date('M j, Y', strtotime($resource['created_at'])); ?></td>
                                                <td>
                                                    <button class="btn btn-sm btn-outline-secondary edit-resource-btn" 
                                                            data-resource-id="<?php echo $resource['id']; ?>"
                                                            data-resource-data='<?php echo json_encode($resource); ?>'
                                                            data-bs-toggle="modal" data-bs-target="#editResourceModal">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                    <form method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to delete this resource? This action cannot be undone.')">
                                                        <input type="hidden" name="resource_id" value="<?php echo $resource['id']; ?>">
                                                        <button type="submit" name="delete_resource" class="btn btn-sm btn-outline-danger">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </form>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <!-- Add Resource Modal -->
    <div class="modal fade" id="addResourceModal" tabindex="-1" aria-labelledby="addResourceModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addResourceModalLabel">Add New Educational Resource</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST" enctype="multipart/form-data">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-12 mb-3">
                                <label for="title" class="form-label">Title</label>
                                <input type="text" class="form-control" name="title" required>
                            </div>
                            <div class="col-md-12 mb-3">
                                <label for="content" class="form-label">Description</label>
                                <textarea class="form-control" id="content" name="content" rows="3"></textarea>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="category" class="form-label">Category</label>
                                <input type="text" class="form-control" name="category" placeholder="e.g., Nutrition, Exercise" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="resource_type" class="form-label">Resource Type</label>
                                <select class="form-select" name="resource_type" required>
                                    <option value="Video">Video</option>
                                    <option value="PDF">PDF</option>
                                    <option value="Audio">Audio</option>
                                </select>
                            </div>
                            <div class="col-md-12 mb-3">
                                <label for="resource_file" class="form-label">File</label>
                                <input type="file" class="form-control" name="resource_file" accept=".pdf,.mp4,.webm,.ogg,.mp3,.wav" required>
                                <small class="form-text text-muted">Upload a video, PDF, or audio file.</small>
                            </div>
                        </div>
                        <div class="mb-3 form-check">
                            <input type="checkbox" class="form-check-input" name="is_active" checked>
                            <label class="form-check-label" for="is_active">Active (Visible to patients)</label>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="add_resource" class="btn btn-primary">Upload Resource</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    <!-- Edit Resource Modal -->
    <div class="modal fade" id="editResourceModal" tabindex="-1" aria-labelledby="editResourceModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="editResourceModalLabel">Edit Educational Resource</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST" enctype="multipart/form-data">
                    <div class="modal-body">
                        <input type="hidden" id="edit_resource_id" name="resource_id">
                        <div class="row">
                            <div class="col-md-12 mb-3">
                                <label for="edit_title" class="form-label">Title</label>
                                <input type="text" class="form-control" id="edit_title" name="title" required>
                            </div>
                            <div class="col-md-12 mb-3">
                                <label for="edit_content" class="form-label">Description</label>
                                <textarea class="form-control" id="edit_content" name="content" rows="3"></textarea>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="edit_category" class="form-label">Category</label>
                                <input type="text" class="form-control" id="edit_category" name="category" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="edit_resource_type" class="form-label">Resource Type</label>
                                <select class="form-select" id="edit_resource_type" name="resource_type" required>
                                    <option value="Video">Video</option>
                                    <option value="PDF">PDF</option>
                                    <option value="Audio">Audio</option>
                                </select>
                            </div>
                            <div class="col-md-12 mb-3">
                                <label for="edit_resource_file" class="form-label">Upload New File (Optional)</label>
                                <input type="file" class="form-control" name="resource_file" accept=".pdf,.mp4,.webm,.ogg,.mp3,.wav">
                                <small class="form-text text-muted">If you upload a new file, it will replace the current one.</small>
                            </div>
                        </div>
                        <div class="mb-3 form-check">
                            <input type="checkbox" class="form-check-input" id="edit_is_active" name="is_active">
                            <label class="form-check-label" for="edit_is_active">Active (Visible to patients)</label>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="update_resource" class="btn btn-primary">Update Resource</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Edit resource modal population
        document.querySelectorAll('.edit-resource-btn').forEach(button => {
            button.addEventListener('click', function() {
                const resourceData = JSON.parse(this.dataset.resourceData);
                
                document.getElementById('edit_resource_id').value = resourceData.id;
                document.getElementById('edit_title').value = resourceData.title;
                document.getElementById('edit_content').value = resourceData.content;
                document.getElementById('edit_category').value = resourceData.category;
                document.getElementById('edit_resource_type').value = resourceData.resource_type;
                document.getElementById('edit_is_active').checked = resourceData.is_active == 1;
            });
        });
    </script>
</body>
</html>