<?php
require_once '../config.php';
requireRole('doctor');

// The safeHtmlspecialchars function is now in config.php
// function safeHtmlspecialchars($value, $default = '') { return htmlspecialchars($value ?? $default, ENT_QUOTES, 'UTF-8'); }

$db = new Database();
$conn = $db->getConnection();

// Get doctor info
$stmt = $conn->prepare("SELECT * FROM doctors WHERE user_id = ?");
$stmt->execute([$_SESSION['user_id']]);
$doctor = $stmt->fetch();

// Get all treatment plans
$stmt = $conn->query("
    SELECT tp.*, p.name as patient_name, p.age, p.gender
    FROM treatment_plans tp
    JOIN patients p ON tp.patient_id = p.id
    ORDER BY tp.created_at DESC
");
$plans = $stmt->fetchAll();

// Handle form submission for new treatment plan
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $patient_id = (int)$_POST['patient_id'];
    $diagnosis = sanitize($_POST['diagnosis']);
    $treatment_type = sanitize($_POST['treatment_type']);
    $medication_name = sanitize($_POST['medication_name']);
    $dosage = sanitize($_POST['dosage']);
    $frequency = sanitize($_POST['frequency']);
    $start_date = $_POST['start_date'];
    $end_date = $_POST['end_date'];
    $instructions = sanitize($_POST['instructions']);
    
    try {
        $stmt = $conn->prepare("INSERT INTO treatment_plans 
            (patient_id, doctor_id, diagnosis, treatment_type, medication_name, dosage, frequency, start_date, end_date, instructions)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        
        $stmt->execute([
            $patient_id, $doctor['id'], $diagnosis, $treatment_type, $medication_name, 
            $dosage, $frequency, $start_date, $end_date, $instructions
        ]);
        
        $success = 'Treatment plan created successfully!';
        logActivity($_SESSION['user_id'], 'Created treatment plan', "Patient ID: $patient_id");
        
        // Refresh the page to show the new plan
        header('Location: treatment_plans.php');
        exit();
    } catch (Exception $e) {
        $error = 'Failed to create treatment plan: ' . $e->getMessage();
    }
}

// Get all patients
$stmt = $conn->query("SELECT * FROM patients ORDER BY name");
$patients = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Treatment Plans - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .status-badge {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        .status-active { background-color: #198754; color: white; }
        .status-completed { background-color: #6c757d; color: white; }
        .status-cancelled { background-color: #dc3545; color: white; }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Treatment Plans</h1>
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addPlanModal">
                        <i class="fas fa-plus me-2"></i>New Treatment Plan
                    </button>
                </div>
                
                <?php if (isset($success)): ?>
                    <div class="alert alert-success alert-dismissible fade show">
                        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if (isset($error)): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <!-- Treatment Plans Table -->
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0">All Treatment Plans</h6>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive-lg">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Patient</th>
                                        <th>Diagnosis</th>
                                        <th>Treatment Type</th>
                                        <th>Medication</th>
                                        <th>Start Date</th>
                                        <th>End Date</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($plans as $plan): 
                                        $status_class = '';
                                        if ($plan['status'] == 'active') $status_class = 'status-active';
                                        elseif ($plan['status'] == 'completed') $status_class = 'status-completed';
                                        elseif ($plan['status'] == 'cancelled') $status_class = 'status-cancelled';
                                    ?>
                                        <tr>
                                            <td>
                                                <div><?php echo safeHtmlspecialchars($plan['patient_name']); ?></div>
                                                <small class="text-muted"><?php echo $plan['age']; ?>y, <?php echo ucfirst($plan['gender']); ?></small>
                                            </td>
                                            <td><?php echo nl2br(safeHtmlspecialchars(substr($plan['diagnosis'], 0, 50) . (strlen($plan['diagnosis'] ?? '') > 50 ? '...' : ''))); ?></td>
                                            <td><?php echo ucfirst($plan['treatment_type']); ?></td>
                                            <td><?php echo safeHtmlspecialchars($plan['medication_name'], 'N/A'); ?></td>
                                            <td><?php echo date('M j, Y', strtotime($plan['start_date'])); ?></td>
                                            <td><?php echo $plan['end_date'] ? date('M j, Y', strtotime($plan['end_date'])) : 'N/A'; ?></td>
                                            <td>
                                                <span class="status-badge <?php echo $status_class; ?>">
                                                    <?php echo ucfirst($plan['status']); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <a href="view_treatment.php?id=<?php echo $plan['id']; ?>" class="btn btn-sm btn-outline-primary">
                                                    <i class="fas fa-eye"></i> View
                                                </a>
                                                <a href="edit_treatment.php?id=<?php echo $plan['id']; ?>" class="btn btn-sm btn-outline-secondary">
                                                    <i class="fas fa-edit"></i> Edit
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <!-- Add Treatment Plan Modal -->
    <div class="modal fade" id="addPlanModal" tabindex="-1" aria-labelledby="addPlanModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addPlanModalLabel">Create New Treatment Plan</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="patient_id" class="form-label">Patient *</label>
                                <select class="form-select" id="patient_id" name="patient_id" required>
                                    <option value="">Select Patient</option>
                                    <?php foreach ($patients as $patient): ?>
                                        <option value="<?php echo $patient['id']; ?>">
                                            <?php echo safeHtmlspecialchars($patient['name']) . ' (Age: ' . safeHtmlspecialchars($patient['age']) . ')'; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="treatment_type" class="form-label">Treatment Type *</label>
                                <select class="form-select" id="treatment_type" name="treatment_type" required>
                                    <option value="">Select Type</option>
                                    <option value="medication">Medication</option>
                                    <option value="lifestyle">Lifestyle Modification</option>
                                    <option value="surgical">Surgical Intervention</option>
                                    <option value="other">Other</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="diagnosis" class="form-label">Diagnosis *</label>
                            <textarea class="form-control" id="diagnosis" name="diagnosis" rows="3" required placeholder="Enter diagnosis..."></textarea>
                        </div>
                        
                        <div id="medicationFields">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="medication_name" class="form-label">Medication Name</label>
                                    <input type="text" class="form-control" id="medication_name" name="medication_name" placeholder="e.g., Metformin, Clomiphene">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label for="dosage" class="form-label">Dosage</label>
                                    <input type="text" class="form-control" id="dosage" name="dosage" placeholder="e.g., 500mg">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label for="frequency" class="form-label">Frequency</label>
                                    <input type="text" class="form-control" id="frequency" name="frequency" placeholder="e.g., Twice daily">
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="start_date" class="form-label">Start Date *</label>
                                <input type="date" class="form-control" id="start_date" name="start_date" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="end_date" class="form-label">End Date (Optional)</label>
                                <input type="date" class="form-control" id="end_date" name="end_date">
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="instructions" class="form-label">Instructions *</label>
                            <textarea class="form-control" id="instructions" name="instructions" rows="3" required placeholder="Enter detailed instructions for the patient..."></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Create Plan</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Show/hide medication fields based on treatment type
        document.getElementById('treatment_type').addEventListener('change', function() {
            const medicationFields = document.getElementById('medicationFields');
            if (this.value === 'medication') {
                medicationFields.style.display = 'block';
            } else {
                medicationFields.style.display = 'none';
            }
        });
        
        // Initialize the display state
        document.getElementById('medicationFields').style.display = 'none';
    </script>
</body>
</html>