<?php
// view_comprehensive_assessment.php
require_once '../config.php';
requireRole('doctor');

$db = new Database();
$conn = $db->getConnection();

// Get assessment ID from URL
$assessment_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$assessment_id) {
    header("Location: pcos_analysis.php");
    exit();
}

// Fetch comprehensive assessment details - UPDATED QUERY
$stmt = $conn->prepare("
    SELECT 
        pca.*,
        p.name as patient_name,
        p.age,
        p.gender,
        p.contact,
        p.medical_history,
        p.created_at as patient_since
    FROM pcos_comprehensive_assessments pca
    JOIN patients p ON pca.patient_id = p.id
    WHERE pca.id = ?
");
$stmt->execute([$assessment_id]);
$assessment = $stmt->fetch();

if (!$assessment) {
    header("Location: pcos_analysis.php?error=Assessment not found");
    exit();
}

// Fetch patient's previous assessments for comparison
$stmt = $conn->prepare("
    SELECT id, created_at, pcos_probability, risk_level, bmi, hirsutism_score
    FROM pcos_comprehensive_assessments 
    WHERE patient_id = ? AND id != ?
    ORDER BY created_at DESC 
    LIMIT 5
");
$stmt->execute([$assessment['patient_id'], $assessment_id]);
$previous_assessments = $stmt->fetchAll();

// Calculate trends if previous assessments exist
$trends = [];
if (!empty($previous_assessments)) {
    $latest_previous = $previous_assessments[0];
    $trends['probability_change'] = $assessment['pcos_probability'] - $latest_previous['pcos_probability'];
    $trends['bmi_change'] = $assessment['bmi'] - $latest_previous['bmi'];
    $trends['hirsutism_change'] = $assessment['hirsutism_score'] - $latest_previous['hirsutism_score'];
}

// Fetch recommendations for this assessment
$stmt = $conn->prepare("
    SELECT * FROM pcos_recommendations 
    WHERE assessment_id = ? 
    ORDER BY priority_level DESC, created_at DESC
");
$stmt->execute([$assessment_id]);
$recommendations = $stmt->fetchAll();

// Handle recommendation actions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['update_recommendation'])) {
        $rec_id = (int)$_POST['recommendation_id'];
        $status = sanitize($_POST['status']);
        $doctor_notes = sanitize($_POST['doctor_notes']);
        
        $stmt = $conn->prepare("
            UPDATE pcos_recommendations 
            SET status = ?, doctor_notes = ?, updated_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$status, $doctor_notes, $rec_id]);
        
        $success = "Recommendation updated successfully.";
    }
    
    if (isset($_POST['add_recommendation'])) {
        $description = sanitize($_POST['description']);
        $priority = sanitize($_POST['priority']);
        $category = sanitize($_POST['category']);
        
        $stmt = $conn->prepare("
            INSERT INTO pcos_recommendations 
            (assessment_id, patient_id, description, priority_level, category, status, created_at)
            VALUES (?, ?, ?, ?, ?, 'pending', NOW())
        ");
        $stmt->execute([$assessment_id, $assessment['patient_id'], $description, $priority, $category]);
        
        $success = "New recommendation added successfully.";
        
        // Refresh recommendations
        $stmt = $conn->prepare("SELECT * FROM pcos_recommendations WHERE assessment_id = ? ORDER BY priority_level DESC");
        $stmt->execute([$assessment_id]);
        $recommendations = $stmt->fetchAll();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PCOS Assessment Details - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .assessment-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 15px;
            padding: 2rem;
            margin-bottom: 2rem;
        }
        
        .metric-card {
            background: white;
            border-radius: 10px;
            padding: 1.5rem;
            margin-bottom: 1rem;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            border-left: 4px solid;
        }
        
        .metric-card.primary { border-left-color: #667eea; }
        .metric-card.success { border-left-color: #28a745; }
        .metric-card.warning { border-left-color: #ffc107; }
        .metric-card.danger { border-left-color: #dc3545; }
        
        .probability-circle {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 2rem;
            font-weight: bold;
            color: white;
            margin: 0 auto;
        }
        
        .prob-high { background: linear-gradient(45deg, #dc3545, #fd7e14); }
        .prob-moderate { background: linear-gradient(45deg, #fd7e14, #ffc107); }
        .prob-low { background: linear-gradient(45deg, #198754, #20c997); }
        
        .criteria-met { background-color: #d4edda; border-left-color: #28a745; }
        .criteria-not-met { background-color: #f8d7da; border-left-color: #dc3545; }
        
        .trend-up { color: #dc3545; }
        .trend-down { color: #28a745; }
        .trend-stable { color: #6c757d; }
        
        .section-card {
            margin-bottom: 2rem;
            border: none;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .recommendation-urgent { border-left: 4px solid #dc3545; }
        .recommendation-high { border-left: 4px solid #fd7e14; }
        .recommendation-medium { border-left: 4px solid #ffc107; }
        .recommendation-low { border-left: 4px solid #6c757d; }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-4">
                <!-- Header -->
                <div class="assessment-header">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h2><i class="fas fa-file-medical me-2"></i>PCOS Assessment Details</h2>
                            <p class="mb-0">Comprehensive diagnostic evaluation for <?php echo htmlspecialchars($assessment['patient_name']); ?></p>
                        </div>
                        <div class="col-md-4 text-end">
                            <a href="pcos_analysis.php" class="btn btn-light me-2">
                                <i class="fas fa-arrow-left me-1"></i>Back to Dashboard
                            </a>
                            <button onclick="window.print()" class="btn btn-light">
                                <i class="fas fa-print me-1"></i>Print
                            </button>
                        </div>
                    </div>
                </div>
                
                <?php if (isset($success)): ?>
                    <div class="alert alert-success alert-dismissible fade show">
                        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <!-- Patient and Assessment Overview -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="metric-card primary text-center">
                            <h5>PCOS Probability</h5>
                            <div class="probability-circle <?php 
                                echo $assessment['pcos_probability'] >= 70 ? 'prob-high' : 
                                    ($assessment['pcos_probability'] >= 40 ? 'prob-moderate' : 'prob-low'); 
                            ?>">
                                <?php echo $assessment['pcos_probability']; ?>%
                            </div>
                            <?php if (isset($trends['probability_change'])): ?>
                                <small class="<?php echo $trends['probability_change'] > 0 ? 'trend-up' : ($trends['probability_change'] < 0 ? 'trend-down' : 'trend-stable'); ?>">
                                    <i class="fas fa-arrow-<?php echo $trends['probability_change'] > 0 ? 'up' : ($trends['probability_change'] < 0 ? 'down' : 'right'); ?>"></i>
                                    <?php echo abs(round($trends['probability_change'], 1)); ?>%
                                </small>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="metric-card success text-center">
                            <h5>Risk Level</h5>
                            <div class="display-4 fw-bold text-<?php 
                                echo $assessment['risk_level'] == 'High' ? 'danger' : 
                                    (strpos($assessment['risk_level'], 'Moderate') !== false ? 'warning' : 'success');
                            ?>">
                                <?php echo $assessment['risk_level']; ?>
                            </div>
                            <p class="mb-0"><?php echo $assessment['diagnosis_category']; ?></p>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="metric-card warning text-center">
                            <h5>BMI</h5>
                            <div class="display-4 fw-bold text-info">
                                <?php echo $assessment['bmi']; ?>
                            </div>
                            <?php if (isset($trends['bmi_change'])): ?>
                                <small class="<?php echo $trends['bmi_change'] > 0 ? 'trend-up' : ($trends['bmi_change'] < 0 ? 'trend-down' : 'trend-stable'); ?>">
                                    <i class="fas fa-arrow-<?php echo $trends['bmi_change'] > 0 ? 'up' : ($trends['bmi_change'] < 0 ? 'down' : 'right'); ?>"></i>
                                    <?php echo abs(round($trends['bmi_change'], 1)); ?>
                                </small>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="metric-card danger text-center">
                            <h5>Rotterdam Score</h5>
                            <div class="display-4 fw-bold text-primary">
                                <?php echo $assessment['rotterdam_score']; ?>/6
                            </div>
                            <p class="mb-0">Criteria Met: <?php 
                                $criteria_count = 0;
                                if ($assessment['menstrual_irregularities'] == 'yes') $criteria_count++;
                                if ($assessment['hirsutism_score'] >= 8) $criteria_count++;
                                if (($assessment['follicle_count_right'] + $assessment['follicle_count_left']) >= 20) $criteria_count++;
                                echo $criteria_count; 
                            ?>/3</p>
                        </div>
                    </div>
                </div>
                
                <!-- Patient Information -->
                <div class="card section-card">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="fas fa-user me-2"></i>Patient Information</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <table class="table table-borderless">
                                    <tr>
                                        <th width="30%">Name:</th>
                                        <td><?php echo htmlspecialchars($assessment['patient_name']); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Age:</th>
                                        <td><?php echo $assessment['age']; ?> years</td>
                                    </tr>
                                    <tr>
                                        <th>Gender:</th>
                                        <td><?php echo $assessment['gender'] ?? 'Not specified'; ?></td>
                                    </tr>
                                    <tr>
                                        <th>Contact:</th>
                                        <td><?php echo htmlspecialchars($assessment['contact']); ?></td>
                                    </tr>
                                </table>
                            </div>
                            <div class="col-md-6">
                                <table class="table table-borderless">
                                    <tr>
                                        <th width="30%">Assessment Date:</th>
                                        <td><?php echo date('F j, Y g:i A', strtotime($assessment['created_at'])); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Patient Since:</th>
                                        <td><?php echo date('F j, Y', strtotime($assessment['patient_since'])); ?></td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                        
                        <?php if (!empty($assessment['medical_history'])): ?>
                        <div class="mt-3">
                            <h6>Medical History:</h6>
                            <p class="text-muted"><?php echo htmlspecialchars($assessment['medical_history']); ?></p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Detailed Assessment Data -->
                <div class="row">
                    <!-- Left Column -->
                    <div class="col-lg-6">
                        <!-- Clinical History -->
                        <div class="card section-card">
                            <div class="card-header"><i class="fas fa-history me-2"></i>Clinical History</div>
                            <div class="card-body">
                                <ul class="list-group list-group-flush">
                                    <li class="list-group-item d-flex justify-content-between"><span>Irregular Cycles:</span> <strong><?php echo ucfirst(safeHtmlspecialchars($assessment['menstrual_irregularities'])); ?></strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>Cycle Length:</span> <strong><?php echo safeHtmlspecialchars($assessment['cycle_length']); ?> days</strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>Oligomenorrhea:</span> <strong><?php echo ucfirst(safeHtmlspecialchars($assessment['oligomenorrhea'])); ?></strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>Amenorrhea:</span> <strong><?php echo ucfirst(safeHtmlspecialchars($assessment['amenorrhea'])); ?></strong></li>
                                </ul>
                            </div>
                        </div>

                        <!-- Laboratory Values -->
                        <div class="card section-card">
                            <div class="card-header"><i class="fas fa-flask me-2"></i>Laboratory Values</div>
                            <div class="card-body">
                                <ul class="list-group list-group-flush">
                                    <li class="list-group-item d-flex justify-content-between"><span>Total Testosterone:</span> <strong><?php echo safeHtmlspecialchars($assessment['total_testosterone']); ?> ng/dL</strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>Free Testosterone:</span> <strong><?php echo safeHtmlspecialchars($assessment['free_testosterone']); ?> pg/mL</strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>DHEA-S:</span> <strong><?php echo safeHtmlspecialchars($assessment['dhea_s']); ?> μg/dL</strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>LH/FSH Ratio:</span> <strong><?php echo safeHtmlspecialchars($assessment['lh_fsh_ratio']); ?></strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>HbA1c:</span> <strong><?php echo safeHtmlspecialchars($assessment['hba1c']); ?>%</strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>Fasting Glucose:</span> <strong><?php echo safeHtmlspecialchars($assessment['fasting_glucose']); ?> mg/dL</strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>Fasting Insulin:</span> <strong><?php echo safeHtmlspecialchars($assessment['fasting_insulin']); ?> μU/mL</strong></li>
                                </ul>
                            </div>
                        </div>
                    </div>

                    <!-- Right Column -->
                    <div class="col-lg-6">
                        <!-- Physical Signs -->
                        <div class="card section-card">
                            <div class="card-header"><i class="fas fa-user-md me-2"></i>Physical Signs</div>
                            <div class="card-body">
                                <ul class="list-group list-group-flush">
                                    <li class="list-group-item d-flex justify-content-between"><span>Hirsutism Score:</span> <strong><?php echo safeHtmlspecialchars($assessment['hirsutism_score']); ?></strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>Acne Severity:</span> <strong><?php echo ucfirst(safeHtmlspecialchars($assessment['acne_severity'])); ?></strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>Androgenic Alopecia:</span> <strong><?php echo ucfirst(safeHtmlspecialchars($assessment['androgenic_alopecia'])); ?></strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>Acanthosis Nigricans:</span> <strong><?php echo ucfirst(safeHtmlspecialchars($assessment['acanthosis_nigricans'])); ?></strong></li>
                                </ul>
                            </div>
                        </div>

                        <!-- Ultrasound Findings -->
                        <div class="card section-card">
                            <div class="card-header"><i class="fas fa-image me-2"></i>Ultrasound Findings</div>
                            <div class="card-body">
                                <ul class="list-group list-group-flush">
                                    <li class="list-group-item d-flex justify-content-between"><span>Ovarian Volume (Avg):</span> <strong><?php echo safeHtmlspecialchars($assessment['ovarian_volume']); ?> mL</strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>Follicles (Right):</span> <strong><?php echo safeHtmlspecialchars($assessment['follicle_count_right']); ?></strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>Follicles (Left):</span> <strong><?php echo safeHtmlspecialchars($assessment['follicle_count_left']); ?></strong></li>
                                    <li class="list-group-item d-flex justify-content-between"><span>Ovarian Stroma:</span> <strong><?php echo ucfirst(safeHtmlspecialchars($assessment['ovarian_stroma'])); ?></strong></li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Recommendations Section -->
                <div class="card section-card">
                    <div class="card-header bg-info text-white">
                        <h5 class="mb-0 d-flex justify-content-between align-items-center">
                            <span><i class="fas fa-tasks me-2"></i>Recommendations</span>
                            <button class="btn btn-sm btn-light" data-bs-toggle="modal" data-bs-target="#addRecommendationModal">
                                <i class="fas fa-plus me-1"></i> Add
                            </button>
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($recommendations)): ?>
                            <p class="text-muted text-center">No recommendations have been added for this assessment.</p>
                        <?php else: ?>
                            <div class="list-group">
                                <?php foreach ($recommendations as $rec): ?>
                                    <div class="list-group-item recommendation-<?php echo strtolower($rec['priority_level']); ?>">
                                        <div class="d-flex w-100 justify-content-between">
                                            <p class="mb-1"><?php echo htmlspecialchars($rec['description']); ?></p>
                                            <small>
                                                <span class="badge bg-<?php 
                                                    if ($rec['status'] == 'completed') echo 'success';
                                                    elseif ($rec['status'] == 'in_progress') echo 'info';
                                                    else echo 'secondary';
                                                ?>"><?php echo ucfirst(str_replace('_', ' ', $rec['status'])); ?></span>
                                            </small>
                                        </div>
                                        <small class="text-muted">
                                            Priority: <?php echo ucfirst($rec['priority_level']); ?> | 
                                            Category: <?php echo ucfirst($rec['category']); ?>
                                        </small>
                                        <?php if ($rec['doctor_notes']): ?>
                                            <p class="mt-2 mb-0 fst-italic text-muted">
                                                <i class="fas fa-comment-medical me-1"></i>
                                                <?php echo htmlspecialchars($rec['doctor_notes']); ?>
                                            </p>
                                        <?php endif; ?>
                                        <div class="mt-2">
                                            <button class="btn btn-sm btn-outline-primary" 
                                                    data-bs-toggle="modal" 
                                                    data-bs-target="#editRecommendationModal"
                                                    onclick="editRecommendation(<?php echo $rec['id']; ?>, '<?php echo $rec['status']; ?>', '<?php echo htmlspecialchars($rec['doctor_notes']); ?>')">
                                                <i class="fas fa-edit me-1"></i> Update Status
                                            </button>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
            </main>
        </div>
    </div>
    
    <!-- Add Recommendation Modal -->
    <div class="modal fade" id="addRecommendationModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <div class="modal-header">
                        <h5 class="modal-title">Add New Recommendation</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Description</label>
                            <textarea class="form-control" name="description" rows="3" required></textarea>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Priority Level</label>
                            <select class="form-select" name="priority" required>
                                <option value="low">Low</option>
                                <option value="medium">Medium</option>
                                <option value="high">High</option>
                                <option value="urgent">Urgent</option>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Category</label>
                            <select class="form-select" name="category" required>
                                <option value="lifestyle">Lifestyle</option>
                                <option value="medication">Medication</option>
                                <option value="monitoring">Monitoring</option>
                                <option value="referral">Referral</option>
                                <option value="education">Patient Education</option>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="add_recommendation" class="btn btn-primary">Add Recommendation</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Edit Recommendation Modal -->
    <div class="modal fade" id="editRecommendationModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <div class="modal-header">
                        <h5 class="modal-title">Edit Recommendation</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <input type="hidden" id="edit_recommendation_id" name="recommendation_id">
                        <div class="mb-3">
                            <label class="form-label">Status</label>
                            <select class="form-select" id="edit_status" name="status" required>
                                <option value="pending">Pending</option>
                                <option value="in_progress">In Progress</option>
                                <option value="completed">Completed</option>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Doctor Notes</label>
                            <textarea class="form-control" id="edit_doctor_notes" name="doctor_notes" rows="3"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="update_recommendation" class="btn btn-primary">Update</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function editRecommendation(id, status, notes) {
            document.getElementById('edit_recommendation_id').value = id;
            document.getElementById('edit_status').value = status;
            document.getElementById('edit_doctor_notes').value = notes;
        }
    </script>
</body>
</html>