<?php
require_once '../config.php';
requireRole('doctor');

$db = new Database();
$conn = $db->getConnection();

$plan_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$plan_id) {
    header('Location: treatment_plans.php');
    exit;
}

// Fetch the treatment plan with patient and doctor details
$stmt = $conn->prepare("
    SELECT 
        tp.*, 
        p.name as patient_name, 
        p.age as patient_age,
        d.name as doctor_name
    FROM treatment_plans tp
    JOIN patients p ON tp.patient_id = p.id
    JOIN doctors d ON tp.doctor_id = d.id
    WHERE tp.id = ?
");
$stmt->execute([$plan_id]);
$plan = $stmt->fetch();

if (!$plan) {
    die('Treatment plan not found.');
}

function getStatusBadgeClass($status) {
    switch ($status) {
        case 'active': return 'bg-success';
        case 'completed': return 'bg-secondary';
        case 'cancelled': return 'bg-danger';
        default: return 'bg-light text-dark';
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>View Treatment Plan - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .plan-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem;
            border-radius: 15px;
            margin-bottom: 2rem;
        }
        .detail-card {
            border-left: 4px solid #667eea;
        }
        .detail-card .card-header {
            background-color: #f8f9fa;
            font-weight: 600;
        }
        @media print {
            body { background-color: white !important; }
            .sidebar, .navbar, .btn-toolbar { display: none !important; }
            main { width: 100% !important; margin: 0 !important; padding: 0 !important; }
            .plan-header { background: none !important; color: black !important; padding: 0 !important; border-bottom: 2px solid #dee2e6; margin-bottom: 1.5rem; border-radius: 0; }
            .card { box-shadow: none !important; border: 1px solid #dee2e6; }
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="plan-header mt-4">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-0">Treatment Plan Details</h1>
                            <p class="mb-0">For Patient: <strong><?php echo safeHtmlspecialchars($plan['patient_name']); ?></strong></p>
                        </div>
                        <div class="btn-toolbar">
                            <a href="treatment_plans.php" class="btn btn-light me-2"><i class="fas fa-arrow-left me-1"></i> Back</a>
                            <a href="edit_treatment.php?id=<?php echo $plan['id']; ?>" class="btn btn-light me-2"><i class="fas fa-edit me-1"></i> Edit</a>
                            <button onclick="window.print()" class="btn btn-light"><i class="fas fa-print me-1"></i> Print</button>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <!-- Left Column -->
                    <div class="col-lg-7">
                        <div class="card detail-card mb-4">
                            <div class="card-header">
                                <i class="fas fa-file-medical me-2"></i>Diagnosis & Plan
                            </div>
                            <div class="card-body">
                                <h5>Diagnosis</h5>
                                <p class="text-muted"><?php echo nl2br(safeHtmlspecialchars($plan['diagnosis'])); ?></p>
                                <hr>
                                <h5>Instructions</h5>
                                <p class="text-muted"><?php echo nl2br(safeHtmlspecialchars($plan['instructions'])); ?></p>
                            </div>
                        </div>

                        <?php if ($plan['treatment_type'] === 'medication' && !empty($plan['medication_name'])): ?>
                        <div class="card detail-card mb-4">
                            <div class="card-header">
                                <i class="fas fa-pills me-2"></i>Medication Details
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <strong>Medication:</strong>
                                        <p><?php echo safeHtmlspecialchars($plan['medication_name']); ?></p>
                                    </div>
                                    <div class="col-md-3">
                                        <strong>Dosage:</strong>
                                        <p><?php echo safeHtmlspecialchars($plan['dosage']); ?></p>
                                    </div>
                                    <div class="col-md-3">
                                        <strong>Frequency:</strong>
                                        <p><?php echo safeHtmlspecialchars($plan['frequency']); ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>

                    <!-- Right Column -->
                    <div class="col-lg-5">
                        <div class="card mb-4">
                            <div class="card-header">
                                <i class="fas fa-info-circle me-2"></i>Plan Overview
                            </div>
                            <ul class="list-group list-group-flush">
                                <li class="list-group-item d-flex justify-content-between">
                                    <strong>Patient:</strong>
                                    <span><?php echo safeHtmlspecialchars($plan['patient_name']); ?> (<?php echo $plan['patient_age']; ?> yrs)</span>
                                </li>
                                <li class="list-group-item d-flex justify-content-between">
                                    <strong>Prescribing Doctor:</strong>
                                    <span><?php echo safeHtmlspecialchars($plan['doctor_name']); ?></span>
                                </li>
                                <li class="list-group-item d-flex justify-content-between">
                                    <strong>Status:</strong>
                                    <span class="badge <?php echo getStatusBadgeClass($plan['status']); ?>"><?php echo ucfirst($plan['status']); ?></span>
                                </li>
                                <li class="list-group-item d-flex justify-content-between">
                                    <strong>Treatment Type:</strong>
                                    <span><?php echo ucfirst($plan['treatment_type']); ?></span>
                                </li>
                                <li class="list-group-item d-flex justify-content-between">
                                    <strong>Start Date:</strong>
                                    <span><?php echo date('F j, Y', strtotime($plan['start_date'])); ?></span>
                                </li>
                                <li class="list-group-item d-flex justify-content-between">
                                    <strong>End Date:</strong>
                                    <span><?php echo $plan['end_date'] ? date('F j, Y', strtotime($plan['end_date'])) : 'Ongoing'; ?></span>
                                </li>
                                <li class="list-group-item d-flex justify-content-between">
                                    <strong>Created On:</strong>
                                    <span><?php echo date('F j, Y', strtotime($plan['created_at'])); ?></span>
                                </li>
                            </ul>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>